package user_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/resource/redisCli"
	"github.com/gin-gonic/gin"
	"hilo-user/_const/redis_key/user_k"
	"hilo-user/domain/model/group_m"
	"hilo-user/domain/model/tim_m"
	"hilo-user/domain/service/user_s"
	"hilo-user/req"
	"hilo-user/resp"
	"time"
)

// @Tags 用户
// @Summary 获取用户详细信息
// @Param token header string true "token"
// @Param timestamp header string true "时间戳"
// @Param nonce header string true "随机数字"
// @Param signature header string true "sha1加密结果"
// @Param deviceType header string true "系统类型 ios android"
// @Param deviceVersion header string true "系统版本"
// @Success 200 {object} user_cv.CvUserDetail
// @Router /v1/user/detail [get]
func UserDetail(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, lang, err := req.GetUserIdLang(c, myContext)
	if err != nil {
		return myContext, err
	}

	cvUserDetail, err := user_s.NewUserService(myContext).GetUserDetail(userId, userId, lang)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, cvUserDetail)
	return myContext, nil
}

// @Tags 用户
// @Summary 获取用户详细信息
// @Param token header string true "token"
// @Param timestamp header string true "时间戳"
// @Param nonce header string true "随机数字"
// @Param signature header string true "sha1加密结果"
// @Param deviceType header string true "系统类型 ios android"
// @Param deviceVersion header string true "系统版本"
// @Param userExternalId path string true "userExternalId"
// @Param groupId query string false "群组id，当传了该id，则返回该用户在该群组的身份"
// @Success 200 {object} user_cv.CvUserDetail
// @Router /v1/user/detail/{userExternalId} [get]
func UserDetailByExternalId(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, lang, err := req.GetUserIdLang(c, myContext)
	if err != nil {
		return myContext, err
	}
	otherUserId, err := req.ToUserId(myContext, mysql.Str(c.Param("userExternalId")))
	if err != nil {
		return nil, err
	}
	model := domain.CreateModelContext(myContext)

	imGroupId := c.Query("groupId")
	if imGroupId != "" {
		imGroupId, err = group_m.ToImGroupId(model, imGroupId)
		if err != nil {
			return myContext, err
		}
	}

	cvUserDetail, err := user_s.NewUserService(myContext).GetUserDetail(otherUserId, userId, lang)
	if err != nil {
		return myContext, err
	}
	if imGroupId != "" {
		cvUserDetail.GroupRole, err = group_m.GetGroupRoleById(model, imGroupId, otherUserId)
		if err != nil {
			return myContext, err
		}
	}

	if cvUserDetail != nil {
		// 检查是否需要同步
		if n, err := redisCli.GetRedis().Exists(model, user_k.GetKeySyncTimHilo(userId)).Result(); err == nil {
			if n == 0 {
				// FIXME：转异步执行
				err = tim_m.FlushHiloInfo(*cvUserDetail.ExternalId, cvUserDetail.IsVip, cvUserDetail.IsPrettyCode,
					nil, cvUserDetail.MyGroupPowerName, cvUserDetail.Noble.Level)
				if err == nil {
					redisCli.GetRedis().Set(model, user_k.GetKeySyncTimHilo(userId), "1", time.Minute)
				} else {
					model.Log.Info("UserBaseByExternalId, FlushHiloInfo failed: ", err)
				}
			} else {
				model.Log.Info("UserDetailByExternalId, no need to sync yet: ", userId)
			}
		} else {
			model.Log.Info("UserDetailByExternalId, check KeySyncTimHilo failed", err)
		}
	}

	resp.ResponseOk(c, cvUserDetail)
	return myContext, nil
}
