package user_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/gin-gonic/gin"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/cv/cp_cv"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/model/bag_m"
	"hilo-user/domain/model/cp_m"
	"hilo-user/domain/model/noble_m"
	"hilo-user/domain/model/res_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/resp"
	"time"
)

type MGetUserLevelReq struct {
	Ids []mysql.ID `form:"ids" binding:"required"`
}

// @Tags 用户-内部
// @Summary 批量获取用户等级
// @Param ids query string true "用户id,如:ids=1&ids=2&ids=3"
// @Success 200 {object} user_cv.MGetUserLevelData
// @Router /inner/user/levels [get]
func MGetUserLevels(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myCtx)
	var req MGetUserLevelReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myCtx, err
	}
	wealthGrade, err := user_m.MGetWealthGrade(model, req.Ids)
	if err != nil {
		return myCtx, err
	}
	charmGrade, err := user_m.MGetCharmGrade(model, req.Ids)
	activeGrade, err := user_m.MGetActiveGrade(model, req.Ids)
	nobleLevel, err := noble_m.BatchGetNobleLevel(model, req.Ids)
	response := user_cv.MGetUserLevelData{}
	for _, userId := range req.Ids {
		response[userId] = user_cv.CvUserLevel{
			UserId:          userId,
			WealthUserGrade: wealthGrade[userId],
			CharmUserGrade:  charmGrade[userId],
			ActiveUserGrade: activeGrade[userId],
			NobleLevel:      nobleLevel[userId],
		}
	}

	resp.ResponseOk(c, response)
	return myCtx, nil
}

type GetUserBagReq struct {
	BagId mysql.ID `form:"bagId" binding:"required"`
}

// @Tags 用户-内部
// @Summary 获取单个背包
// @Param bagId query int true "背包id"
// @Success 200 {object} user_cv.UserBag
// @Router /inner/user/bag/id [get]
func GetUserBagId(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myCtx)
	var req GetUserBagReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myCtx, err
	}
	bag, err := bag_m.GetUserBag(model, req.BagId)
	if err != nil {
		return myCtx, err
	}
	gift, err := res_m.FindResGift(model, bag.ResId)
	if err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, user_cv.UserBag{
		BagId:      bag.ID,
		ResType:    bag.ResType,
		ResId:      bag.ResId,
		GiftId:     bag.ResId,
		Name:       gift.Name,
		DiamondNum: gift.DiamondNum,
		IconUrl:    gift.IconUrl,
		SvgaUrl:    gift.SvagUrl,
		Count:      bag.Count,
		RemainDays: int(bag.EndTime.Sub(time.Now()).Hours() / 24),
	})
	return myCtx, nil
}

type GetUserCpReq struct {
	Id mysql.ID `form:"id" binding:"required"`
}

// @Tags 用户-内部
// @Summary 获取用户cp
// @Param id query int true "用户id"
// @Success 200 {object} cp_cv.CvCp
// @Router /inner/user/cp [get]
func GetUserCp(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myContext)
	var req GetUserCpReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myContext, err
	}
	userId := req.Id
	var response cp_cv.CvCp
	cpLevel := cp_e.CpLevel0
	cpRelation, exists := cp_m.GetCpRelation(model, userId)
	if !exists {
		resp.ResponseOk(c, response)
		return myContext, nil
	}
	var myPrivilegeList []cp_cv.CvPrivilege
	level := cp_m.GetCpLevel(model, cpRelation.ID)
	if level.ExpireAt.Before(time.Now()) {
		level.ExpireAt = time.Now().AddDate(0, 1, 0)
	}
	cpLevel = level.Level
	cpUserId := cpRelation.UserId2
	if cpUserId == userId {
		cpUserId = cpRelation.UserId1
	}
	userPrivileges, err := cp_m.MGetUserSvipPrivilege(model, []uint64{userId})
	if err != nil {
		return myContext, err
	}
	privilegeList := cp_cv.CopyCpLevelPrivilegeList(level.Level, "en")
	for i, v := range privilegeList {
		if v.CanSwitch {
			privilegeList[i].UserSwitch = userPrivileges[userId][v.Type]
		}
	}
	// 我的特权,有开关并且打开才返回
	for i, v := range privilegeList {
		if !v.CanSwitch {
			myPrivilegeList = append(myPrivilegeList, privilegeList[i])
		} else if v.UserSwitch {
			myPrivilegeList = append(myPrivilegeList, privilegeList[i])
		}
	}
	userBases, err := user_cv.GetUserBaseMap([]uint64{cpUserId}, userId)
	if err != nil {
		return myContext, err
	}
	// 返回值
	response = cp_cv.CvCp{
		CpUserInfo: userBases[cpUserId],
		CpLevel: cp_cv.CvCpLevel{
			Level: cpLevel,
		},
		MyPrivilegeList: myPrivilegeList,
		CpDays:          int(time.Now().Sub(cpRelation.CreatedTime).Hours()/24) + 1,
		CreatedUnix:     cpRelation.CreatedTime.Unix(),
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

// @Tags 用户-内部
// @Summary 获取用户cp关系
// @Param id query int true "用户id"
// @Success 200 {object} cp_cv.CvCpRelation
// @Router /inner/user/cpRelation [get]
func GetUserCpRelation(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myContext)
	var req GetUserCpReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myContext, err
	}
	userId := req.Id
	var response cp_cv.CvCpRelation
	cpRelation, exists := cp_m.GetCpRelation(model, userId)
	if !exists {
		resp.ResponseOk(c, response)
		return myContext, nil
	}
	cpUserId := cpRelation.UserId2
	if cpUserId == userId {
		cpUserId = cpRelation.UserId1
	}
	response = cp_cv.CvCpRelation{
		CpId:     cpRelation.ID,
		UserId:   userId,
		CpUserId: cpUserId,
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

type GetUserCpPairReq struct {
	Ids []mysql.ID `form:"ids" binding:"required"`
}

// @Tags 用户-内部
// @Summary 给出指定uids下的cp对
// @Param ids query string true "用户id,如:ids=1&ids=2&ids=3"
// @Success 200 {object} [][]uint64
// @Router /inner/user/cp/pair [get]
func GetUserCpPair(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myContext)
	var req GetUserCpPairReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myContext, err
	}
	userIds := req.Ids
	m := make(map[mysql.ID]bool)
	for _, uid := range userIds {
		m[uid] = true
	}
	pairs := cp_m.MGetCpRelation(model, userIds)
	var response [][2]uint64
	for _, pair := range pairs {
		if m[pair.UserId1] && m[pair.UserId2] {
			response = append(response, [2]mysql.ID{pair.UserId1, pair.UserId2})
		}
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}
