package user_r

import (
	"errors"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/gin-gonic/gin"
	"hilo-user/domain/model/bag_m"
	"hilo-user/domain/service/headwear_s"
	"hilo-user/domain/service/noble_s"
	"hilo-user/domain/service/ride_s"
	"hilo-user/resp"
)

type SendUserBagReq struct {
	UserId  mysql.ID   `form:"userId" binding:"required"`
	ResType mysql.Type `form:"resType" binding:"required"` // 道具类型 1:礼物道具
	ResId   mysql.ID   `form:"resId" binding:"required"`   // 道具资源id
	Count   mysql.Num  `form:"count" binding:"min=1"`      // 下发数量
	Day     int        `form:"day" binding:"min=1"`        // 天数
	Reason  string     `form:"reason" binding:"required"`  // 原因
}

// @Tags 用户-内部
// @Summary 发送背包道具
// @Param userId formData int true "用户id"
// @Param resType formData int true "道具类型 1:礼物道具"
// @Param resId formData int true "道具资源id"
// @Param count formData int true "下发数量"
// @Param day formData int true "天数"
// @Param reason formData string true "原因"
// @Success 200
// @Router /inner/prop/bag/send [post]
func SendUserBag(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myCtx)
	var req SendUserBagReq
	if err := c.ShouldBind(&req); err != nil {
		return myCtx, err
	}
	//var bagId mysql.ID
	var err error
	switch req.ResType {
	case mysql.Type(1):
		if _, err = bag_m.AddUserBag(model, req.UserId, req.ResType, req.ResId, req.Count, req.Day, req.Reason); err != nil {
			return myCtx, err
		}
	default:
		return myCtx, errors.New("un support type")
	}
	//userBag, err := bag_m.GetUserBag(model, bagId)
	//if err != nil {
	//	return myCtx, err
	//}
	resp.ResponseOk(c, struct{}{})
	return myCtx, nil
}

type SendUserNobleReq struct {
	UserId mysql.ID `form:"userId" binding:"required"`
	Level  uint16   `form:"level" binding:"min=1,max=5"`
	Day    int      `form:"day" binding:"min=1"`
}

// @Tags 用户-内部
// @Summary 发送用户贵族
// @Param userId formData int true "用户id"
// @Param level formData int true "贵族等级"
// @Param day formData int true "下发天数"
// @Success 200
// @Router /inner/prop/noble/send [post]
func SendUserNoble(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var req SendUserNobleReq
	if err := c.ShouldBind(&req); err != nil {
		return myCtx, err
	}
	if err := noble_s.NewNobleService(myCtx).SendNoble(req.UserId, req.Level, req.Day); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, struct{}{})
	return myCtx, nil
}

type SendUserHeadwearReq struct {
	UserId     mysql.ID `form:"userId" binding:"required"`
	HeadwearId mysql.ID `form:"headwearId" binding:"required"`
	Day        int      `form:"day" binding:"min=1"`
}

// @Tags 用户-内部
// @Summary 发送用户头饰
// @Param userId formData int true "用户id"
// @Param headwearId formData int true "头饰id"
// @Param day formData int true "天数"
// @Success 200
// @Router /inner/prop/headwear/send [post]
func SendUserHeadwear(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var req SendUserHeadwearReq
	if err := c.ShouldBind(&req); err != nil {
		return myCtx, err
	}
	if err := headwear_s.NewHeadwearService(myCtx).SendHeadwear(req.UserId, req.HeadwearId, req.Day); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, struct{}{})
	return myCtx, nil
}

type SendUserRideReq struct {
	UserId mysql.ID `form:"userId" binding:"required"`
	RideId mysql.ID `form:"rideId" binding:"required"`
	Day    int      `form:"day" binding:"min=1"`
}

// @Tags 用户-内部
// @Summary 发送用户座驾
// @Param userId formData int true "用户id"
// @Param rideId formData int true "座驾id"
// @Param day formData int true "天数"
// @Success 200
// @Router /inner/prop/ride/send [post]
func SendUserRide(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var req SendUserRideReq
	if err := c.ShouldBind(&req); err != nil {
		return myCtx, err
	}
	if err := ride_s.NewRideService(myCtx).SendRide(req.UserId, req.RideId, req.Day); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, struct{}{})
	return myCtx, nil
}
