package user_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"gorm.io/gorm"
	"hilo-user/_const/enum/headwear_e"
	"hilo-user/domain/model/res_m"
	"hilo-user/myerr"
	"strconv"
	"time"
)

//用户头饰
type UserHeadwear struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	UserId        mysql.ID
	HeadwearId    mysql.ID
	EndTime       time.Time
	Using         headwear_e.UserHeadwearUsing
}

//获取用户头饰，不存在则新建
func GetUserHeadwearOrInit(model *domain.Model, userId mysql.ID, headwearId mysql.ID) (*UserHeadwear, error) {
	//检查headwearId是否存在/有效
	if flag, err := res_m.CheckHeadwearValidById(model, headwearId); err != nil {
		return nil, err
	} else {
		if flag == false {
			return nil, myerr.NewSysError("headwearId： " + strconv.FormatUint(headwearId, 10) + " 无效")
		}
	}

	userHeadwear := UserHeadwear{}
	if err := model.Db.Model(&UserHeadwear{}).Where(&UserHeadwear{
		UserId:     userId,
		HeadwearId: headwearId,
	}).First(&userHeadwear).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return &UserHeadwear{
				Model:      model,
				UserId:     userId,
				HeadwearId: headwearId,
				EndTime:    time.Now(),
			}, nil
		} else {
			return nil, myerr.WrapErr(err)
		}
	}
	userHeadwear.Model = model
	return &userHeadwear, nil
}

//设置为使用中
func (userHeadwear *UserHeadwear) SetUsing() (*UserHeadwear, error) {
	if err := resetAllUserHeadwearNoUsing(userHeadwear.Model, userHeadwear.UserId); err != nil {
		return nil, err
	}
	userHeadwear.Using = headwear_e.YesUsing
	return userHeadwear, nil
}

//增加结束时间
func (userHeadwear *UserHeadwear) AddEndTime(t headwear_e.UserHeadwearLogOrginType, second uint32, operateUserId mysql.ID) (*UserHeadwear, mysql.ID, error) {
	logId, err := addUserHeadwearLog(userHeadwear.Model, userHeadwear.UserId, userHeadwear.HeadwearId, t, headwear_e.AddSecond, &second, nil, operateUserId)
	if err != nil {
		return nil, 0, myerr.WrapErr(err)
	}
	//if err := resetAllUserHeadwearNoUsing(userHeadwear.Model, userHeadwear.UserId); err != nil {
	//	return nil, logId, err
	//}
	nowTime := time.Now()
	if userHeadwear.EndTime.After(nowTime) {
		nowTime = userHeadwear.EndTime
	}
	userHeadwear.EndTime = nowTime.Add(time.Duration(second) * time.Second)
	return userHeadwear, logId, nil
}

//删除
func (userHeadwear *UserHeadwear) MgrDel(mgrId mysql.ID) (*UserHeadwear, mysql.ID, error) {
	logId, err := addUserHeadwearLog(userHeadwear.Model, userHeadwear.UserId, userHeadwear.HeadwearId, headwear_e.Mgr, headwear_e.Del, nil, nil, mgrId)
	if err != nil {
		return nil, 0, myerr.WrapErr(err)
	}
	userHeadwear.SetDel()
	return userHeadwear, logId, nil
}

//重置所有的座驾均为不使用状态
func resetAllUserHeadwearNoUsing(model *domain.Model, userId mysql.ID) error {
	if err := model.Db.Model(&UserHeadwear{}).Where(&UserHeadwear{
		UserId: userId,
	}).UpdateColumn("using", headwear_e.NoUsing).Error; err != nil {
		return myerr.WrapErr(err)
	}
	return nil
}

//用户装饰日志
type UserHeadwearLog struct {
	mysql.Entity
	UserId        mysql.ID
	OperateUserId mysql.ID
	HeadwearId    mysql.ID
	OriginType    headwear_e.UserHeadwearLogOrginType
	Type          headwear_e.UserHeadwearLogType
	AddSecond     *mysql.Num
	UpdateEndTime *time.Time
}

//增加修改日志
func addUserHeadwearLog(model *domain.Model, userId mysql.ID, headwearId mysql.ID, originType headwear_e.UserHeadwearLogOrginType, t headwear_e.UserHeadwearLogType, addSecond *uint32, UpdateEndTime *time.Time, operateUserId mysql.ID) (mysql.ID, error) {
	userHeadwearLog := UserHeadwearLog{
		UserId:        userId,
		OperateUserId: operateUserId,
		HeadwearId:    headwearId,
		OriginType:    originType,
		Type:          t,
		AddSecond:     addSecond,
		UpdateEndTime: UpdateEndTime,
	}
	if err := model.Db.Create(&userHeadwearLog).Error; err != nil {
		return 0, myerr.WrapErr(err)
	}
	return userHeadwearLog.ID, nil
}

func GetUserHeadwearUsing(model *domain.Model, userId mysql.ID) (*UserHeadwear, error) {
	userHeadwear := UserHeadwear{}
	if err := model.Db.Model(&UserHeadwear{}).Where(&UserHeadwear{
		UserId: userId,
	}).Where("end_time > ?", time.Now()).Order("`using` desc, updated_time desc").First(&userHeadwear).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		} else {
			return nil, myerr.WrapErr(err)
		}
	}
	userHeadwear.Model = model
	return &userHeadwear, nil
}
