package cp_r

import (
	"fmt"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"github.com/gin-gonic/gin"
	"github.com/jinzhu/now"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/cv/cp_cv"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/model/cp_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/myerr/bizerr"
	"hilo-user/req"
	"hilo-user/resp"
	"time"
)

// @Tags CP v2
// @Summary 排行榜
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param pageIndex query int true "偏移值 默认：1" default(1)
// @Param pageSize query int true "请求数量 默认：10" default(10)
// @Param queryType path string true "类型：day/week/month"
// @Success 200 {object} []cp_cv.CvCpRank
// @Router /v2/cp/rank/{queryType} [get]
func CpRank(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	pageReq := new(req.PageReqBase)
	if err := c.ShouldBindQuery(pageReq); err != nil {
		return myCtx, err
	}
	if pageReq.PageIndex == 0 {
		pageReq.PageIndex = 1
	}
	appVersion, deviceType := c.GetHeader(mycontext.APP_VERSION), c.GetHeader(mycontext.DEVICETYPE)
	if deviceType == "android" {
		if lower, _ := utils.CompareVersion(appVersion, fmt.Sprintf("<= %s", "3.9.0")); lower {
			pageReq.PageSize = 30
		}
	}
	queryType := c.Param("queryType")
	if queryType != "day" && queryType != "week" && queryType != "month" {
		return myCtx, bizerr.InvalidParameter
	}
	var beginDate, endDate string
	switch queryType {
	case "day":
		beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
	case "week":
		beginDate = now.BeginningOfWeek().Format("2006-01-02")
		endDate = now.EndOfWeek().Format("2006-01-02")
	case "month":
		beginDate = now.BeginningOfMonth().Format("2006-01-02")
		endDate = now.EndOfMonth().Format("2006-01-02")
	}
	offset, limit := (pageReq.PageIndex-1)*pageReq.PageSize, pageReq.PageSize
	model := domain.CreateModelContext(myCtx)
	ranks := cp_m.PageCpDayRank(model, beginDate, endDate, offset, limit)
	var response []cp_cv.CvCpRank
	var userIds []mysql.ID
	var cpIds []mysql.ID
	for _, rank := range ranks {
		userIds = append(userIds, rank.UserId1)
		userIds = append(userIds, rank.UserId2)
		cpIds = append(cpIds, rank.CpId)
	}
	userBase, err := user_cv.GetUserBaseMap(userIds, userId)
	if err != nil {
		return myCtx, err
	}
	cpLevels := cp_m.MGetCpLevel(model, cpIds)
	for i, rank := range ranks {
		response = append(response, cp_cv.CvCpRank{
			CpId:    rank.CpId,
			User1:   user_cv.UserBaseToUserLittle(userBase[rank.UserId1]),
			User2:   user_cv.UserBaseToUserLittle(userBase[rank.UserId2]),
			Score:   rank.Score,
			Ranking: fmt.Sprintf("%d", i+1+offset),
			CpLevel: cp_cv.CvCpLevel{
				Level: cpLevels[rank.CpId].Level,
				//Points:       cp_e.CpLevelPoints[cpLevel] + curPoints,
				//EndPoints:   nextPoints,
				//RemainPoints: remainPoints,
				//ExpireAtUnix: expireAtUnix,
			},
		})
	}
	resp.ResponsePageBaseOk(c, response, pageReq.PageIndex+1, len(ranks) >= pageReq.PageSize)
	return myCtx, nil
}

// @Tags CP v2
// @Summary CP,Top3
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Success 200 {object} cp_cv.CpTops
// @Router /v2/cp/top3 [get]
func CpTop3(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	var response = cp_cv.CpTops{
		Day:  make([]cp_cv.CvCpRank, 0),
		Week: make([]cp_cv.CvCpRank, 0),
	}
	queryTypes := []string{"day", "week"}
	for _, queryType := range queryTypes {
		var beginDate, endDate string
		switch queryType {
		case "day":
			beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
		case "week":
			beginDate = now.BeginningOfWeek().Format("2006-01-02")
			endDate = now.EndOfWeek().Format("2006-01-02")
		}
		offset, limit := 0, 3
		model := domain.CreateModelContext(myCtx)
		ranks := cp_m.PageCpDayRank(model, beginDate, endDate, offset, limit)
		var userIds []mysql.ID
		var cpIds []mysql.ID
		for _, rank := range ranks {
			userIds = append(userIds, rank.UserId1)
			userIds = append(userIds, rank.UserId2)
			cpIds = append(cpIds, rank.CpId)
		}
		userBase, err := user_cv.GetUserBaseMap(userIds, userId)
		if err != nil {
			return myCtx, err
		}
		cpLevels := cp_m.MGetCpLevel(model, cpIds)
		for i, rank := range ranks {
			if queryType == "day" {
				response.Day = append(response.Day, cp_cv.CvCpRank{
					CpId:    rank.CpId,
					User1:   user_cv.UserBaseToUserLittle(userBase[rank.UserId1]),
					User2:   user_cv.UserBaseToUserLittle(userBase[rank.UserId2]),
					Score:   rank.Score,
					Ranking: fmt.Sprintf("%d", i+1+offset),
					CpLevel: cp_cv.CvCpLevel{
						Level: cpLevels[rank.CpId].Level,
					},
				})
			} else {
				response.Week = append(response.Week, cp_cv.CvCpRank{
					CpId:    rank.CpId,
					User1:   user_cv.UserBaseToUserLittle(userBase[rank.UserId1]),
					User2:   user_cv.UserBaseToUserLittle(userBase[rank.UserId2]),
					Score:   rank.Score,
					Ranking: fmt.Sprintf("%d", i+1+offset),
					CpLevel: cp_cv.CvCpLevel{
						Level: cpLevels[rank.CpId].Level,
					},
				})
			}
		}
	}
	resp.ResponseOk(c, response)
	return myCtx, nil
}

// @Tags CP v2
// @Summary 我的cp
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param queryType path string true "类型：day/week/month"
// @Success 200 {object} cp_cv.CvCpRank
// @Router /v2/cp/my/{queryType} [get]
func CpMy(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	queryType := c.Param("queryType")
	if queryType != "day" && queryType != "week" && queryType != "month" {
		return myCtx, bizerr.InvalidParameter
	}
	var beginDate, endDate string
	switch queryType {
	case "day":
		beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
	case "week":
		beginDate = now.BeginningOfWeek().Format("2006-01-02")
		endDate = now.EndOfWeek().Format("2006-01-02")
	case "month":
		beginDate = now.BeginningOfMonth().Format("2006-01-02")
		endDate = now.EndOfMonth().Format("2006-01-02")
	}
	model := domain.CreateModelContext(myCtx)
	relation, exits := cp_m.GetCpRelation(model, userId)
	var userIds []mysql.ID
	var scores uint32
	if exits {
		// 保证一下自己是userId1
		userIds = append(userIds, userId)
		if relation.UserId1 == userId {
			userIds = append(userIds, relation.UserId2)
		} else {
			userIds = append(userIds, relation.UserId1)
		}
		rank := cp_m.GetCpDayRank(model, beginDate, endDate, relation.Id)
		scores = rank.Score
	} else {
		userIds = append(userIds, userId)
		relation.UserId1 = userId
	}
	userBases, err := user_cv.GetUserBaseMap(userIds, userId)
	if err != nil {
		return myCtx, err
	}
	response := cp_cv.CvCpRank{
		CpId:  relation.Id,
		Score: scores,
	}
	if relation.UserId1 > 0 {
		response.User1 = user_cv.UserBaseToUserLittle(userBases[relation.UserId1])
	}
	if relation.UserId2 > 0 {
		response.User2 = user_cv.UserBaseToUserLittle(userBases[relation.UserId2])
		response.Ranking = "30+"
		ranks := cp_m.PageCpDayRank(model, beginDate, endDate, 0, 30)
		for i, rank := range ranks {
			if relation.Id == rank.CpId {
				response.Ranking = fmt.Sprintf("%d", i+1)
				break
			}
		}
	}
	resp.ResponseOk(c, response)
	return myCtx, nil
}

// @Tags CP v2
// @Summary 成就页
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param pageIndex query int true "偏移值 默认：1" default(1)
// @Param pageSize query int true "请求数量 默认：10" default(10)
// @Success 200 {object} []cp_cv.CvCpAchievement
// @Router /v2/cp/achievement [get]
func CpAchievement(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	model := domain.CreateModelContext(myCtx)
	_, lang, err := req.GetUserIdLang(c, myCtx)
	if err != nil {
		return myCtx, err
	}
	var response = make([]cp_cv.CvCpAchievement, 0)
	TypeDescMap := map[cp_e.CpAchievement]uint{
		cp_e.CpAchievementLevel:     280, //"等级分值最高",
		cp_e.CpAchievementVisitors:  281, //"空间访问量最高",
		cp_e.CpAchievementMonthRank: 282, //"月榜最高",
		cp_e.CpAchievementWeekRank:  283, //"周榜最高",
		cp_e.CpAchievementDayRank:   284, //"天榜最高",
	}
	achievements := cp_m.GetCpAchievements(model)
	var userIds []uint64
	var cpIds []uint64
	for _, v := range achievements {
		userIds = append(userIds, v.UserId1)
		userIds = append(userIds, v.UserId2)
		cpIds = append(cpIds, v.CpId)
	}
	users, err := user_m.GetUserMapByIds(model, userIds)
	if err != nil {
		return myCtx, err
	}
	cpLevels := cp_m.MGetCpLevel(model, cpIds)
	for _, a := range achievements {
		response = append(response, cp_cv.CvCpAchievement{
			CpId:     a.CpId,
			User1:    user_cv.UserToTiny(users[a.UserId1]),
			User2:    user_cv.UserToTiny(users[a.UserId2]),
			CpLevel:  cpLevels[a.CpId].Level,
			Type:     a.Type,
			TypeDesc: cp_m.GetTranslate(TypeDescMap[a.Type], lang),
			Score:    a.Score,
			TimeUnix: a.UpdatedTime.Unix(),
		})
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myCtx, nil
}
