package invite_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/utils"
	"github.com/gin-gonic/gin"
	"github.com/jinzhu/now"
	"hilo-user/cv/invite_cv"
	"hilo-user/domain/cache/user_c"
	"hilo-user/domain/model/invite_m"
	"hilo-user/domain/model/promotion_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/myerr/bizerr"
	"hilo-user/req"
	"hilo-user/resp"
	"time"
)

// @Tags 新人邀请
// @Summary 提交申请
// @Param newUserCode formData string true "被邀请人id"
// @Param platform formData string true "平台"
// @Param platformId formData string true "平台Id"
// @Param recharge formData string true "充值金额"
// @Param userCode formData string true "邀请人id"
// @Param videoUrl formData string true "充值金额"
// @Success 200
// @Router /v2/user/invite/apply [post]
func InviteApply(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)

	type paramStr struct {
		NewUserCode string `form:"newUserCode" binding:"required"`
		Platform    string `form:"platform" binding:"required"`
		PlatformId  string `form:"platformId" binding:"required"`
		Recharge    string `form:"recharge" binding:"required"`
		UserCode    string `form:"userCode" binding:"required"`
		VideoUrl    string `form:"videoUrl" binding:"required"`
	}

	myUserId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	model := domain.CreateModelContext(myCtx)
	// 每日最多提交10次
	noLimitTimesMap := map[uint64]struct{}{1770691: {}, 2264431: {}, 7283111: {}}
	if _, ok := noLimitTimesMap[myUserId]; !ok {
		// 今日提交次数
		times, err := invite_m.TodayInviteApplyCount(model, myUserId)
		if err != nil {
			model.Log.Errorf("InviteApply myUserId:%v, err:%v", myUserId, err)
			return myCtx, err
		}
		if times >= 15 {
			return myCtx, bizerr.InviteApplyTimes
		}
	}

	var param paramStr
	if err := c.ShouldBind(&param); err != nil {
		return myCtx, err
	}
	// 平台是否填写正确
	platforms := promotion_m.GetPromotionPlatforms(model)
	var existsPlatform bool
	for _, v := range platforms {
		if v == param.Platform {
			existsPlatform = true
		}
	}
	if !existsPlatform {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, bizerr.InvalidParameter
	}
	// code 是否存在
	newUser, err := user_m.GetUserByCode(model, param.NewUserCode)
	if err != nil {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, bizerr.InviteApplyNewCodeInvalid
	}
	user, err := user_m.GetUserByCode(model, param.UserCode)
	if err != nil {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, bizerr.InviteApplyCodeInvalid
	}
	//if user.ID != myUserId { //只能自己提交
	//	return myCtx, bizerr.InvalidParameter
	//}
	if newUser.ID == 0 || user.ID == 0 {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, bizerr.InvalidParameter
	}
	// 邀请人是否有资格邀请
	if !promotion_m.IsPromotionAgent(model, user.ID) {
		model.Log.Errorf("InviteApply 没有邀请资格 param:%v", param)
		return myCtx, bizerr.InviteApplyNoPermission
	}
	//if user.ID != myUserId && !promotion_m.IsMyPromotionManager(model, user.ID, myUserId) {
	//	model.Log.Errorf("InviteApply 没有邀请资格 param:%v", param)
	//	return myCtx, bizerr.InviteApplyNoPermission
	//}
	// 被邀请人是否已经被人提交过申请
	isApply, err := invite_m.IsInInviteApply(model, newUser.ID)
	if err != nil {
		model.Log.Errorf("InviteApply param:%v, err:%v", param, err)
		return myCtx, err
	}
	if isApply {
		model.Log.Errorf("InviteApply 已经被别人邀请过了 param:%v", param)
		return myCtx, bizerr.InviteApplyAlreadyInvited
	}
	// 被邀请人是否符合条件
	isInvite, err := promotion_m.IsPromotionInvitee(model, newUser.ID)
	if err != nil {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, err
	}
	if isInvite {
		model.Log.Errorf("InviteApply 已经被别人邀请了 param:%v", param)
		return myCtx, bizerr.InviteApplyAlreadyInvited
	}
	// 插入邀请表
	err = invite_m.CreateInviteApply(model, user.ID, newUser.ID, myUserId, param.Platform, param.PlatformId, param.Recharge, param.VideoUrl)
	if err != nil {
		model.Log.Errorf("InviteApply param:%v", param)
		return myCtx, err
	}

	resp.ResponseOk(c, nil)
	return myCtx, nil
}

// @Tags 新人邀请
// @Summary 查询历史申请
// @Param pageIndex query int true "偏移值 默认：1" default(1)
// @Param pageSize query int true "请求数量 默认：10" default(10)
// @Param beginTime query string true "开始时间2006-01-02"
// @Param endTime query string true "结束时间2006-01-02"
// @Param type query int true "1.已申请2.待审核3.已通过4.已拒绝"
// @Success 200 {object} invite_cv.InviteApplyRes
// @Router /v2/user/invite/apply [get]
func InviteApplyList(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)

	type paramStr struct {
		PageIndex int       `form:"pageIndex" binding:"required"`
		PageSize  int       `form:"pageSize" binding:"required"`
		BeginTime time.Time `form:"beginTime" binding:"required" time_format:"2006-01-02"`
		EndTime   time.Time `form:"endTime" binding:"required" time_format:"2006-01-02"`
		Type      int       `form:"type"`
	}

	var param paramStr
	if err := c.ShouldBindQuery(&param); err != nil {
		return myCtx, err
	}
	if param.Type < 0 || param.Type > 4 {
		return myCtx, bizerr.InvalidParameter
	}

	//beginTime, err := time.ParseInLocation(utils.DATE_FORMAT, param.BeginTime, time.Local)
	//if err != nil {
	//	return nil, myerr.WrapErr(err)
	//}
	//endTime, err := time.ParseInLocation(utils.DATE_FORMAT, param.EndTime, time.Local)
	//if err != nil {
	//	return nil, myerr.WrapErr(err)
	//}
	param.EndTime = utils.GetDayEndTime(param.EndTime)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}

	var model = domain.CreateModelContext(myCtx)

	agentIds := []uint64{userId}
	if promotion_m.IsPromotionManager(model, userId) {
		agentIds, err = promotion_m.GetPromotionManagerAgentList(model, userId)
		if err != nil {
			return myCtx, err
		}
	}

	if param.Type == 0 { // 返回所有Type类型有多少条数
		user, err := user_c.GetUserTinyById(model, userId)
		if err != nil {
			model.Log.Errorf("GetApplyList param:%v, err:%v", param, err)
			return myCtx, bizerr.InvalidParameter
		}
		numList := make([]*invite_cv.InviteApplyNumRes, 0, 4)
		for _, gType := range []int{1, 2, 3, 4} { // 1.已申请2.待审核3.已通过4.已拒绝
			num, err := invite_m.GetInviteApplyNumByType(model, gType, param.BeginTime, param.EndTime, agentIds, userId)
			if err != nil {
				return myCtx, err
			}
			numList = append(numList, &invite_cv.InviteApplyNumRes{Type: gType, Num: num})
		}
		resp.ResponsePageOk(c, &invite_cv.InviteApplyRes{NumList: numList, MyCode: user.Code}, 0, 0, 0)
		return myCtx, nil
	}

	list, total, err := invite_m.GetApplyList(model, userId, agentIds, param.PageIndex, param.PageSize, param.Type, param.BeginTime, param.EndTime)
	if err != nil {
		model.Log.Errorf("GetApplyList param:%v, err:%v", param, err)
		return myCtx, err
	}

	uids := make([]uint64, 0, len(list)+1)
	for _, v := range list {
		uids = append(uids, v.UserId, v.NewUserId)
	}
	users, err := user_c.GetUserTinyMap(model, uids, true)
	if err != nil {
		model.Log.Errorf("GetApplyList param:%v, err:%v", param, err)
		return myCtx, err
	}
	res := &invite_cv.InviteApplyRes{}
	res.List = make([]*invite_cv.InviteApply, 0, len(list))
	for _, v := range list {
		res.List = append(res.List, &invite_cv.InviteApply{
			NewUserCode: users[v.NewUserId].Code,
			Platform:    v.Platform,
			Recharge:    v.RechargeInfo,
			UserCode:    users[v.UserId].Code,
			CreateUnix:  v.CreatedTime.Unix(),
			Level:       v.Level,
			Status:      v.Status,
			Avatar:      users[v.NewUserId].Avatar,
			ExternalId:  users[v.NewUserId].ExternalId,
			Reason:      v.Reason,
			Country:     users[v.NewUserId].Country,
			PlatformId:  v.PlatformId,
		})
	}

	resp.ResponsePageOk(c, res, total, param.PageSize, param.PageIndex)
	return myCtx, nil
}

// @tags 新人邀请
// @Summary 平台列表
// @Success 200 {object} []string
// @Router /v2/user/invite/platform [get]
func PromotionPlatform(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	resp.ResponseOk(c, []string{"Falla", "Yalla", "Whisper", "Ahlan", "Mashi", "YoYo", "Yoho", "Echo", "Hawa",
		"Yalla Ludo", "Hafla", "Imo", "Ola Party", "ShareChat", "Viya", "Hello Yo", "Bigo Live", "Hago", "Oye Talk", "Tiktok", "Bigo", "Mr7ba"})
	return myCtx, nil
}

// @Tags 新人邀请
// @Summary 查询周期
// @Success 200 {object} invite_cv.AgentPeriod
// @Router /v2/user/invite/period [get]
func AgentPeriod(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var response invite_cv.AgentPeriod
	week := now.BeginningOfWeek().AddDate(0, 0, 1) // 周一开始
	for i := 0; i < 12; i++ {
		response.Week = append(response.Week, invite_cv.AgentPeriodWeek{
			StartDate: week.Format("2006-01-02"),
			EndDate:   week.AddDate(0, 0, 6).Format("2006-01-02"),
		})
		week = week.AddDate(0, 0, -7)
	}
	resp.ResponseOk(c, response)
	return myCtx, nil
}
