package cp_cv

import (
	"fmt"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/bluele/gcache"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/model/res_m"
	"time"
)

// cp信息
type CvCpInfo struct {
	UserInfo   user_cv.UserTiny `json:"userInfo"`   // 用户信息
	CpUserInfo user_cv.UserTiny `json:"cpUserInfo"` // cp用户信息
	CpDays     int              `json:"cpDays"`     // cp天数
	VisitTimes int              `json:"visitTimes"` // 空间访问量
}

// cp等级
type CvCpLevel struct {
	Level        cp_e.CpLevel `json:"level"`                  // 等级 0:无称号 1:恋爱CP 2:甜蜜CP 3:忠诚CP 4:炽热CP 5:荣耀CP
	Points       uint32       `json:"points"`                 // CP值
	NextPoints   uint32       `json:"nextPoints,omitempty"`   // 下个等级所需CP值
	RemainPoints uint32       `json:"remainPoints,omitempty"` // 还需要多少CP值到下一个级别
	ExpireAtUnix int64        `json:"expireAtUnix,omitempty"` // 有效期,时间戳
}

// 资源等级
type CvResLevel struct {
	Level cp_e.CpLevel `json:"level"` // 等级
	Icon  string       `json:"icon"`  // 等级icon图
}

// 特权信息
type CvPrivilege struct {
	Type       cp_e.CpPrivilege `json:"type"`       // 特权id 1:空间 2:横幅 3:等级勋章 4:证书 5:进场特效 6:头像头饰 7:动态资料卡 8:麦位特效
	NameMsgId  uint             `json:"-"`          // 名称-翻译id
	Name       string           `json:"name"`       // 名称
	Icon       string           `json:"icon"`       // 图标
	CanSwitch  bool             `json:"canSwitch"`  // 能否开关
	UserSwitch bool             `json:"userSwitch"` // 用户开关
}

// cp空间页
type CvSpace struct {
	CpInfo        CvCpInfo      `json:"cpInfo"`        // cp信息
	CpLevel       CvCpLevel     `json:"cpLevel"`       // cp等级
	ResLevelList  []CvResLevel  `json:"resLevelList"`  // 资源等级列表,无称号/恋爱CP/甜蜜CP/忠诚CP/炽热CP/荣耀CP
	PrivilegeList []CvPrivilege `json:"privilegeList"` // 等级特权
}

var CvResLevelList = []CvResLevel{
	{cp_e.CpLevel0, cp_e.CpLevelIcon[cp_e.CpLevel0]},
	{cp_e.CpLevel1, cp_e.CpLevelIcon[cp_e.CpLevel1]},
	{cp_e.CpLevel2, cp_e.CpLevelIcon[cp_e.CpLevel2]},
	{cp_e.CpLevel3, cp_e.CpLevelIcon[cp_e.CpLevel3]},
	{cp_e.CpLevel4, cp_e.CpLevelIcon[cp_e.CpLevel4]},
	{cp_e.CpLevel5, cp_e.CpLevelIcon[cp_e.CpLevel5]},
}

var (
	CvPrivilege1 = CvPrivilege{cp_e.CpPrivilegeSpace, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeSpace], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeSpace], false, false}
	CvPrivilege2 = CvPrivilege{cp_e.CpPrivilegeBanner, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeBanner], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeBanner], false, false}
	CvPrivilege3 = CvPrivilege{cp_e.CpPrivilegeMedal, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeMedal], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeMedal], false, false}
	CvPrivilege4 = CvPrivilege{cp_e.CpPrivilegeCert, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeCert], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeCert], false, false}
	CvPrivilege5 = CvPrivilege{cp_e.CpPrivilegeRoomEffect, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeRoomEffect], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeRoomEffect], true, false}
	CvPrivilege6 = CvPrivilege{cp_e.CpPrivilegeHeadwear, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeHeadwear], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeHeadwear], false, false}
	CvPrivilege7 = CvPrivilege{cp_e.CpPrivilegeActiveProfile, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeActiveProfile], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeActiveProfile], true, false}
	CvPrivilege8 = CvPrivilege{cp_e.CpPrivilegeMicEffect, cp_e.CpPrivilegeNameMsgId[cp_e.CpPrivilegeMicEffect], "", cp_e.CpPrivilegeIcon[cp_e.CpPrivilegeMicEffect], false, false}
)

var CpLevelPrivilegeList = map[cp_e.CpLevel][]CvPrivilege{
	cp_e.CpLevel0: {},
	cp_e.CpLevel1: {CvPrivilege1, CvPrivilege2, CvPrivilege3, CvPrivilege4},
	cp_e.CpLevel2: {CvPrivilege1, CvPrivilege2, CvPrivilege3, CvPrivilege4, CvPrivilege5},
	cp_e.CpLevel3: {CvPrivilege1, CvPrivilege2, CvPrivilege3, CvPrivilege4, CvPrivilege5, CvPrivilege6},
	cp_e.CpLevel4: {CvPrivilege1, CvPrivilege2, CvPrivilege3, CvPrivilege4, CvPrivilege5, CvPrivilege6, CvPrivilege7},
	cp_e.CpLevel5: {CvPrivilege1, CvPrivilege2, CvPrivilege3, CvPrivilege4, CvPrivilege5, CvPrivilege6, CvPrivilege7, CvPrivilege8},
}

func CopyCpLevelPrivilegeList(Level cp_e.CpLevel, lang string) []CvPrivilege {
	privileges := make([]CvPrivilege, len(CpLevelPrivilegeList[Level]))
	copy(privileges, CpLevelPrivilegeList[Level])
	for i, v := range privileges {
		privileges[i].Name = GetTranslate(v.NameMsgId, lang)
	}
	return privileges
}

var translateCache = gcache.New(1000).LRU().Build()

func GetTranslate(msgId uint, lang string) string {
	key := fmt.Sprintf("%v-%v", msgId, lang)
	if data, err := translateCache.Get(key); err == nil {
		return data.(string)
	}
	if resMul, _ := res_m.GetResMultiTextBy(mysql.Db, msgId, lang); resMul != nil {
		_ = translateCache.SetWithExpire(key, resMul.Content, time.Hour)
		return resMul.Content
	}
	return "default"
}
