package cp_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"github.com/gin-gonic/gin"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/cv/cp_cv"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/model/cp_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/myerr/bizerr"
	"hilo-user/req"
	"hilo-user/resp"
	"strconv"
	"time"
)

type PostPutAnniversaryReq struct {
	Content   string `form:"content" binding:"required"`
	Timestamp int64  `form:"timestamp" binding:"required"`
	IsRemind  bool   `form:"isRemind"`
}

// @Tags CP v2
// @Summary 发布纪念日
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param content formData string true "纪念日名称"
// @Param timestamp formData int true "时间戳"
// @Param isRemind formData bool false "是否提醒"
// @Success 200
// @Router /v2/cp/anniversary [post]
func PostAnniversary(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	var param PostPutAnniversaryReq
	if err := c.ShouldBind(&param); err != nil {
		return myCtx, err
	}
	model := domain.CreateModelContext(myCtx)
	relation, exits := cp_m.GetCpRelation(model, userId)
	if !exits {
		return myCtx, bizerr.CpNotRelation
	} else {
		if err := cp_m.AddCpAnniversary(model, cp_e.AnniversaryItemTypeNormal, relation, param.Content, param.Timestamp, param.IsRemind, 0); err != nil {
			return myCtx, err
		}
	}
	resp.ResponseOk(c, "")
	return myCtx, nil
}

// @Tags CP v2
// @Summary 修改纪念日
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param content formData string true "纪念日名称"
// @Param timestamp formData int true "时间戳"
// @Param isRemind formData bool false "是否提醒"
// @Param id path int true "更新的记录id"
// @Success 200
// @Router /v2/cp/anniversary/{id} [put]
func PutAnniversary(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var param PostPutAnniversaryReq
	if err := c.ShouldBind(&param); err != nil {
		return myCtx, err
	}
	id, _ := strconv.ParseUint(c.Param("id"), 10, 64)
	model := domain.CreateModelContext(myCtx)
	if err := cp_m.UpdateCpAnniversary(model, id, param.Content, param.Timestamp, param.IsRemind); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, "")
	return myCtx, nil
}

// @Tags CP v2
// @Summary 获取纪念日
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param pageIndex query int true "偏移值 默认：1" default(1)
// @Param pageSize query int true "请求数量 默认：10" default(10)
// @Success 200 {object} []cp_cv.CvCpAnniversary
// @Router /v2/cp/anniversary [get]
func PageAnniversary(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	var response = make([]cp_cv.CvCpAnniversary, 0)
	model := domain.CreateModelContext(myCtx)
	cpRelation, exits := cp_m.GetCpRelation(model, userId)
	if exits {
		userIds := []uint64{cpRelation.UserId1, cpRelation.UserId2}
		users, err := user_m.GetUserMapByIds(model, userIds)
		if err != nil {
			return myCtx, err
		}
		cpUserId := cpRelation.UserId2
		if cpUserId == userId {
			cpUserId = cpRelation.UserId1
		}
		userInfo := user_cv.UserToTiny(users[userId])
		cpUserInfo := user_cv.UserToTiny(users[cpUserId])
		response = append(response, cp_cv.CvCpAnniversary{
			Type: cp_e.AnniversaryItemTypeAvatar,
			CpInfo: &cp_cv.CvCpBase{
				UserInfo:   *userInfo,
				CpUserInfo: *cpUserInfo,
			},
			OriginTimestamp: cpRelation.CreatedTime.Unix(),
			Timestamp:       cpRelation.CreatedTime.Unix(),
			CanDel:          false,
		})
	}
	anniversary := cp_m.GetAllCpAnniversary(model, userId)
	for _, v := range anniversary {
		timestamp := v.Timestamp
		if v.Type == cp_e.AnniversaryItemTypeAnniversary && timestamp > 0 {
			timestamp = calcNextAnniversary(timestamp)
		}
		// 客户端只认识0 1
		Type := v.Type
		if v.Type != cp_e.AnniversaryItemTypeAvatar {
			Type = cp_e.AnniversaryItemTypeNormal
		}
		response = append(response, cp_cv.CvCpAnniversary{
			Type:            Type,
			Id:              v.ID,
			Content:         v.Content,
			Timestamp:       timestamp,
			OriginTimestamp: v.Timestamp,
			IsRemind:        v.IsRemind,
			CanDel:          true,
			IsTop:           v.Sort > 0,
		})
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myCtx, nil
}

// 计算下一个纪念日
func calcNextAnniversary(timestamp int64) int64 {
	now := time.Now()
	// 还没超过一天,不用计算明年的
	if now.Unix()-timestamp < 86400 {
		return timestamp
	}
	birthday := time.Unix(timestamp, 0)
	// 计算今年的生日日期
	thisYearBirthday := time.Date(now.Year(), birthday.Month(), birthday.Day(), birthday.Hour(), birthday.Minute(), birthday.Second(), 0, time.Local)

	// 如果今年的生日还未到，则生日日期为今年的生日日期；否则为明年的生日日期
	var next time.Time
	if thisYearBirthday.After(now) {
		next = thisYearBirthday
	} else {
		next = thisYearBirthday.AddDate(1, 0, 0)
	}

	// 计算时间戳
	nextTimestamp := next.Unix()
	return nextTimestamp
}

// @Tags CP v2
// @Summary 获取纪念日
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param id path int true "记录id"
// @Success 200
// @Router /v2/cp/anniversary/:id [delete]
func DelAnniversary(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	id, _ := strconv.ParseUint(c.Param("id"), 10, 64)
	model := domain.CreateModelContext(myCtx)
	if err := cp_m.DelCpAnniversary(model, id); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, "")
	return myCtx, nil
}
