package cp_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/utils"
	"gorm.io/gorm"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/domain/model/user_m"
	"hilo-user/myerr/bizerr"
	"time"
)

type CpRelation struct {
	Id          uint64    `json:"id"`
	UserId1     uint64    `json:"userId1"`
	UserId2     uint64    `json:"userId2"`
	CreatedTime time.Time `json:"createdTime"`
}

type CpInvite struct {
	Id           uint64              `json:"id"`
	UserId       uint64              `json:"userId"`
	InviteUserId uint64              `json:"inviteUserId"`
	DiamondNum   uint32              `json:"diamondNum"`
	Status       cp_e.CpInviteStatus `json:"status"`
}

// 发送私信-发起邀请
type CpInviteMessage struct {
	Identifier string           `json:"identifier"`
	Msg        string           `json:"msg"`
	Sender     *user_m.UserTiny `json:"sender"`
}

// 发送私信-接受邀请
type CpAcceptInviteMessage struct {
	Identifier string           `json:"identifier"`
	Msg        string           `json:"msg"`
	Sender     *user_m.UserTiny `json:"sender"`
	Receiver   *user_m.UserTiny `json:"receiver"`
}

// 发送私信-拒绝邀请
type CpDenyInviteMessage struct {
	Identifier string           `json:"identifier"`
	Msg        string           `json:"msg"`
	Sender     *user_m.UserTiny `json:"sender"`
}

type CpCancel struct {
	Id        uint64              `json:"id"`
	UserId    uint64              `json:"userId"`
	RecUserId uint64              `json:"recUserId"`
	Status    cp_e.CpCancelStatus `json:"status"`
}

// 发送私信（解除）
type CpCancelMessage struct {
	Identifier string           `json:"identifier"`
	Msg        string           `json:"msg"`
	Sender     *user_m.UserTiny `json:"sender"`
}

// 发送私信-撤销解除、接受解除
type CpDealCancelMessage struct {
	Identifier string `json:"identifier"`
	Msg        string `json:"msg"`
	Status     uint8  `json:"status"` //1.撤销解除2.接受解除
}

type Cp struct {
	Id               uint64    `json:"id"`
	UserId1          uint64    `json:"userId1"`
	UserId2          uint64    `json:"userId2"`
	DisconnectSecond int64     `json:"disconnectSecond"`
	Score            int32     `json:"score"`
	DayScore         int32     `json:"dayScore"`
	PeriodDay        string    `json:"periodDay"`
	WeekScore        int32     `json:"weekScore"`
	PeriodWeek       string    `json:"periodWeek"`
	MonthScore       int32     `json:"monthScore"`
	PeriodMonth      string    `json:"periodMonth"`
	Status           int8      `json:"status"`
	CreatedTime      time.Time `json:"createdTime"`
}

func CreateCp(model *domain.Model, userId1, userId2 uint64) error {
	userIds := []uint64{userId1, userId2}
	oldCp := make([]*Cp, 0)
	// 这两个人以前是否有旧的cp关系
	err := model.DB().Model(Cp{}).Where("status = 1 and user_id1 in (?) and user_id2 in (?)", userIds, userIds).Find(&oldCp).Error
	if err != nil {
		model.Log.Errorf("CreateCp user1:%d, user2:%d, err:%v", userId1, userId2, err)
		return err
	}
	createdTime := time.Now()
	if len(oldCp) > 0 {
		// 旧的cp关系，有效的时间给他加回去
		oldSecond := time.Duration(time.Now().Unix() - oldCp[0].CreatedTime.Unix() - oldCp[0].DisconnectSecond)
		if oldSecond > 0 {
			createdTime = createdTime.Add(-1 * time.Second * time.Duration(time.Now().Unix()-oldCp[0].CreatedTime.Unix()-oldCp[0].DisconnectSecond))
		}
	}

	result := model.DB().Exec("insert into cp_relation(user_id1, user_id2, created_time) select ?,?,? where not exists (select user_id1 from cp_relation where user_id1 in (?) or user_id2 in (?));", userId1, userId2, createdTime.Format(utils.DATETIME_FORMAT), userIds, userIds)
	if result.Error != nil {
		model.Log.Errorf("CreateCp user1:%d, user2:%d, err:%v", userId1, userId2, result.Error)
		return result.Error
	}
	if result.RowsAffected <= 0 {
		model.Log.Errorf("CreateCp user1:%d, user2:%d, err:%v", userId1, userId2, bizerr.TransactionFailed)
		return bizerr.TransactionFailed
	}
	return nil
}

func GetCp(model *domain.Model, userId uint64) (*CpRelation, error) {
	res := new(CpRelation)
	err := model.DB().Model(CpRelation{}).Where("user_id1 = ? or user_id2 = ?", userId, userId).First(&res).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			return res, nil
		}
		model.Log.Errorf("CreateCp userId:%d, err:%v", userId, err)
		return nil, err
	}
	return res, nil
}

func GetCpInvite(model *domain.Model, userId, userIdInvite uint64, status cp_e.CpInviteStatus) (*CpInvite, error) {
	res := new(CpInvite)
	err := model.DB().Model(CpInvite{}).Where(CpInvite{UserId: userId, InviteUserId: userIdInvite, Status: status}).First(&res).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		}
		model.Log.Errorf("GetCpInvite user1:%d, user2:%d, err:%v", userId, userIdInvite, err)
		return nil, err
	}
	return res, nil
}

func CreateCpInvite(model *domain.Model, userId, userIdInvite uint64, diamondNum uint32) (uint64, error) {
	cpInvite := CpInvite{UserId: userId, InviteUserId: userIdInvite, DiamondNum: diamondNum, Status: cp_e.CpInvite}
	err := model.DB().Model(CpInvite{}).Create(&cpInvite).Error
	if err != nil {
		model.Log.Errorf("CreateCpInvite user1:%d, user2:%d, diamondNum:%d, err:%v", userId, userIdInvite, diamondNum, err)
		return 0, err
	}
	return cpInvite.Id, nil
}

// userId：发起邀请者
func UpdateStatusCpInvite(model *domain.Model, id uint64, status cp_e.CpInviteStatus) error {
	result := model.DB().Exec("update cp_invite set status=? where id=? and status=? limit 1", status, id, cp_e.CpInvite)
	if result.Error != nil {
		model.Log.Errorf("UpdateStatusCpInvite id:%d, status:%d, err:%v", id, status, result.Error)
		return result.Error
	}
	if result.RowsAffected <= 0 {
		model.Log.Errorf("UpdateStatusCpInvite id:%d, status:%d, err:%v", id, status, bizerr.TransactionFailed)
		return bizerr.TransactionFailed
	}
	return nil
}

func CreateCpCancel(model *domain.Model, userId, recUserId uint64) (uint64, error) {
	cpCancel := CpCancel{UserId: userId, RecUserId: recUserId, Status: cp_e.CpCancel}
	err := model.DB().Model(CpCancel{}).Create(&cpCancel).Error
	if err != nil {
		model.Log.Errorf("CreateCpCancel user1:%d, user2:%d, err:%v", userId, recUserId, err)
		return 0, err
	}
	return cpCancel.Id, nil
}

func GetCpCancel(model *domain.Model, userIds []uint64, status cp_e.CpCancelStatus) (*CpCancel, error) {
	res := new(CpCancel)
	err := model.DB().Model(CpCancel{}).Where("status = ? and user_id in (?) and rec_user_id in (?)", status, userIds, userIds).First(&res).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		}
		model.Log.Errorf("GetCpCancel users:%d, err:%v", userIds, err)
		return nil, err
	}
	return res, nil
}

func GetCpCancelWithMe(model *domain.Model, userId uint64, status cp_e.CpCancelStatus) (*CpCancel, error) {
	res := new(CpCancel)
	err := model.DB().Model(CpCancel{}).Where("status = ? and (user_id = ? or rec_user_id = ?)", status, userId, userId).First(&res).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		}
		model.Log.Errorf("GetCpCancel user1:%d, err:%v", userId, err)
		return nil, err
	}
	return res, nil
}

func UpdateStatusCpCancel(model *domain.Model, id uint64, status cp_e.CpCancelStatus) error {
	result := model.DB().Exec("update cp_cancel set status=? where id=? and status=? limit 1", status, id, cp_e.CpCancel)
	if result.Error != nil {
		model.Log.Errorf("UpdateStatusCpCancel id:%d, status:%d, err:%v", id, status, result.Error)
		return result.Error
	}
	if result.RowsAffected <= 0 {
		model.Log.Errorf("UpdateStatusCpCancel id:%d, status:%d, err:%v", id, status, bizerr.TransactionFailed)
		return bizerr.TransactionFailed
	}
	return nil
}

func DelCpRelation(model *domain.Model, userId1, userId2 uint64) error {
	userIds := []uint64{userId1, userId2}
	result := model.DB().Exec("delete from cp_relation where user_id1 in (?) and user_id2 in (?) limit 1;", userIds, userIds)
	if result.Error != nil {
		model.Log.Errorf("DelCpRelation user1:%d, user2:%d, err:%v", userId1, userId2, result.Error)
		return result.Error
	}
	if result.RowsAffected <= 0 {
		model.Log.Errorf("DelCpRelation user1:%d, user2:%d, err:%v", userId1, userId2, bizerr.TransactionFailed)
		return bizerr.TransactionFailed
	}
	return nil
}

func GetCpInviteByTime(model *domain.Model, expTime time.Time) ([]*CpInvite, error) {
	res := make([]*CpInvite, 0)
	err := model.DB().Model(CpInvite{}).Where("status = ? and created_time <= ?", cp_e.CpInvite, expTime.Format(utils.DATETIME_FORMAT)).Find(&res).Error
	if err != nil {
		model.Log.Errorf("GetCpInviteByTime err:%v", err)
		return nil, err
	}
	return res, nil
}

func GetCpCancelByTime(model *domain.Model, expTime time.Time) ([]*CpCancel, error) {
	res := make([]*CpCancel, 0)
	err := model.DB().Model(CpCancel{}).Where("status = ? and created_time <= ?", cp_e.CpCancel, expTime.Format(utils.DATETIME_FORMAT)).Find(&res).Error
	if err != nil {
		model.Log.Errorf("GetCpCancelByTime err:%v", err)
		return nil, err
	}
	return res, nil
}
