package resp

import (
	"encoding/json"
	"git.hilo.cn/hilo-common/mycontext"
	"github.com/gin-gonic/gin"
	"hilo-user/myerr"
	"hilo-user/req"
	"net/http"
)

type Response struct {
	Code             uint16      `json:"code"`             // 错误码
	Message          interface{} `json:"message"`          // 消息
	MessageData      interface{} `json:"messageData"`      // 消息详情
	OperationMessage interface{} `json:"operationMessage"` // 操作消息
	Data             interface{} `json:"data"`             // 数据
}

type GameResponse struct {
	RetCode      uint16      `json:"ret_code"`
	RetMsg       string      `json:"ret_msg"`
	SdkErrorCode uint16      `json:"sdk_error_code"`
	Data         interface{} `json:"data"`
}

/**
 * HTTP输出json信息
 * param: *gin.Context
 * param: error        err
 * param: interface{}  data
 */
func ResponseOk(c *gin.Context, data interface{}) {
	// always return http.StatusOK
	response := Response{
		Code:             myerr.GetSuccessCode(),
		Message:          myerr.GetSuccessMsg(),
		OperationMessage: myerr.GetSuccessMsg(),
		Data:             data,
	}
	printResponseBody(c, &response)

	c.JSON(http.StatusOK, response)
}

// 分页返回
// 客户端入参 req.PageReqBase
// 服务端返回 req.PageRespBase
func ResponsePageBaseOk(c *gin.Context, data interface{}, nextPageIndex int, hasNextPage bool) {
	if data == nil {
		data = make([]interface{}, 0)
	}
	response := Response{
		Code:             myerr.GetSuccessCode(),
		Message:          myerr.GetSuccessMsg(),
		OperationMessage: myerr.GetSuccessMsg(),
		Data: req.PageRespBase{
			NextPageIndex: nextPageIndex,
			HasNextPage:   hasNextPage,
			Data:          data,
		},
	}
	c.JSON(http.StatusOK, response)
}

func ResponseWaring(c *gin.Context, waringError *myerr.WaringError) {
	response := Response{
		Code:             waringError.GetCode(),
		Message:          waringError.GetMsg(),
		OperationMessage: waringError.GetMsg(),
		Data:             nil,
	}
	printResponseBody(c, &response)
	c.JSON(http.StatusOK, response)
}

func ResponseSysError(c *gin.Context, sysError *myerr.SysError) {
	response := Response{
		Code:             sysError.GetCode(),
		Message:          sysError.GetMsg(),
		OperationMessage: sysError.GetMsg(),
		Data:             nil,
	}
	printResponseBody(c, &response)
	c.JSON(http.StatusOK, response)
}

func ResponseBusiness(c *gin.Context, businessError *myerr.BusinessError) {
	response := Response{
		Code:             businessError.GetCode(),
		Message:          businessError.GetMsg(),
		MessageData:      businessError.GetData(),
		OperationMessage: businessError.GetMsg(),
		Data:             nil,
	}
	printResponseBody(c, &response)
	c.JSON(http.StatusOK, response)
}

func ResponseErrWithString(c *gin.Context, msg interface{}) {
	response := Response{
		Code:             myerr.GetSysErrorCode(),
		Message:          msg,
		OperationMessage: msg,
		Data:             nil,
	}
	printResponseBody(c, &response)
	c.JSON(http.StatusOK, response)
}

func printResponseBody(c *gin.Context, response interface{}) {
	traceId, _ := c.Get(mycontext.TRACEID)
	if _traceId, ok := traceId.(string); ok {
		c.Header("X-Trace-ID", _traceId)
	}

	var userId uint64 = 0
	if strUserId, ok := c.Get(mycontext.USERID); ok {
		userId = strUserId.(uint64)
	}

	buf, err := json.Marshal(response)
	body := ""
	if len(buf) < 1024 {
		body = string(buf)
	} else {
		body = string(buf[0:1024])
	}

	if err == nil {
		mycontext.CreateMyContext(c.Keys).Log.Infof("request rsp url:%s, traceId:%v, userId:%d, body:%s", c.Request.RequestURI, traceId, userId, body)
	} else {
		mycontext.CreateMyContext(c.Keys).Log.Error("request rsp body Marshal fail traceId:%v, err:%v", traceId, err)
	}
}
