package user_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"hilo-user/_const/enum/user_e"
	"hilo-user/myerr"
	"hilo-user/myerr/bizerr"
	"time"
)

//用户信息
type User struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	ExternalId    mysql.Str
	Avatar        mysql.Str
	DefaultAvatar bool
	Nick          mysql.Str
	Sex           mysql.Sex
	Birthday      mysql.Timestamp
	Country       mysql.Str
	CountryIcon   mysql.Str
	Language      mysql.Str
	Description   mysql.Str
	Code          mysql.Str
	OriginCode    mysql.Str
	IsPush        mysql.OpenClose
	IsShowAge     mysql.OpenClose
	Status        user_e.UserStatus
	DeviceType    mysql.Str
	LogoutTime    int64
}

type UserTiny struct {
	ID           uint64 `json:"id,omitempty"`
	ExternalId   string `json:"externalId"`
	Avatar       string `json:"avatar"`
	Nick         string `json:"nick"`
	Sex          uint8  `json:"sex"` // 1男2女
	Code         string `json:"code"`
	Description  string `json:"description"`
	Country      string `json:"country"`
	CountryIcon  string `json:"countryIcon"`
	IsPrettyCode bool   `json:"isPrettyCode"` // 是否靓号
	IsLogout     bool   `json:"isLogout"`     //是否注销 true:已经注销， false:没有注销
	//生日，如果是其它人用户信息，年龄则按照是否展示显示，如果是本人，年龄则按照是否存在展示
	Birthday *uint64 `json:"birthday"`
}

func (u User) IsPrettyCode() bool {
	return u.Code != u.OriginCode
}

//获取用户
func GetUser(model *domain.Model, id mysql.ID) (*User, error) {
	var user User
	if err := model.Db.WithContext(model.Context).Where(&User{
		Entity: mysql.Entity{ID: id},
	}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

// 通过code获取用户
func GetUserByCode(model *domain.Model, code string) (*User, error) {
	if code == "" {
		return nil, bizerr.InvalidParameter
	}
	var user User
	if err := model.Db.Where(&User{Code: code}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

//获取用户信息
func GetUsersByIds(model *domain.Model, userIds []mysql.ID) ([]User, error) {
	var users []User
	if err := model.Db.Model(User{}).Where("id IN (?)", userIds).Find(&users).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	return users, nil
}

func GetUserMapByIds(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]User, error) {
	rows, err := GetUsersByIds(model, userIds)
	if err != nil {
		return nil, err
	}
	result := make(map[mysql.ID]User, 0)
	for _, i := range rows {
		result[i.ID] = i
	}
	return result, nil
}

// 通过externalId获取用户
func GetUserByExtId(model *domain.Model, externalId string) (*User, error) {
	var user User
	if err := model.Db.Where(&User{ExternalId: externalId}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

func ToUserTiny(user *User) *UserTiny {
	return &UserTiny{
		ID:           user.ID,
		ExternalId:   user.ExternalId,
		Avatar:       user.Avatar,
		Nick:         user.Nick,
		Sex:          user.Sex,
		Code:         user.Code,
		Description:  user.Description,
		Country:      user.Country,
		CountryIcon:  user.CountryIcon,
		IsPrettyCode: user.IsPrettyCode(),
		IsLogout:     IfLogout(user.LogoutTime),
		Birthday:     BirthdayToUint64(&user.Birthday),
	}
}

func IfLogout(logoutTime int64) bool {
	return logoutTime > 0 && time.Now().Unix() > logoutTime
}

func BirthdayToUint64(birthday *mysql.Timestamp) *uint64 {
	if *birthday == 0 {
		return nil
	}
	return (*uint64)(birthday)
}

//获取用户
func GetUsers(model *domain.Model, ids []mysql.ID) ([]*User, error) {
	res := make([]*User, 0)
	if err := model.Db.WithContext(model.Context).Where("id in (?)", ids).Find(&res).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	return res, nil
}

//获取用户的国家，和所属的区域（是否阿语区）
func GetUserCountryArea(model *domain.Model, userId mysql.ID) (string, int, error) {
	type info struct {
		Name string
		Area int
	}
	res := new(info)
	if err := model.Db.WithContext(model.Context).Raw("select name, area from res_country where name = (select country from user where id = ?)", userId).First(&res).Error; err != nil {
		return "", 0, myerr.WrapErr(err)
	}
	return res.Name, res.Area, nil
}

//获取用户的国家所属的区域（是否阿语区）
func GetUserAreaMap(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]int, error) {
	type info struct {
		Id   mysql.ID
		Area int
	}
	rows := make([]*info, 0)
	sql := "select user.id, rc.area from user left join res_country rc on user.country = rc.name where user.id in (?);"
	if err := model.DB().Raw(sql, userIds).Scan(&rows).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	res := make(map[mysql.ID]int)
	for _, v := range rows {
		res[v.Id] = v.Area
	}
	return res, nil
}
