package user_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/gin-gonic/gin"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/model/bag_m"
	"hilo-user/domain/model/noble_m"
	"hilo-user/domain/model/res_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/resp"
	"time"
)

type MGetUserLevelReq struct {
	Ids []mysql.ID `form:"ids" binding:"required"`
}

// @Tags 用户-内部
// @Summary 批量获取用户等级
// @Param ids query string true "用户id,如:ids=1&ids=2&ids=3"
// @Success 200 {object} user_cv.MGetUserLevelData
// @Router /inner/user/levels [get]
func MGetUserLevels(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myCtx)
	var req MGetUserLevelReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myCtx, err
	}
	wealthGrade, err := user_m.MGetWealthGrade(model, req.Ids)
	if err != nil {
		return myCtx, err
	}
	charmGrade, err := user_m.MGetCharmGrade(model, req.Ids)
	activeGrade, err := user_m.MGetActiveGrade(model, req.Ids)
	nobleLevel, err := noble_m.BatchGetNobleLevel(model, req.Ids)
	response := user_cv.MGetUserLevelData{}
	for _, userId := range req.Ids {
		response[userId] = user_cv.CvUserLevel{
			UserId:          userId,
			WealthUserGrade: wealthGrade[userId],
			CharmUserGrade:  charmGrade[userId],
			ActiveUserGrade: activeGrade[userId],
			NobleLevel:      nobleLevel[userId],
		}
	}

	resp.ResponseOk(c, response)
	return myCtx, nil
}

type GetUserBagReq struct {
	BagId mysql.ID `form:"bagId" binding:"required"`
}

// @Tags 用户-内部
// @Summary 获取单个背包
// @Param bagId query int true "背包id"
// @Success 200 {object} user_cv.UserBag
// @Router /inner/user/bag/id [get]
func GetUserBagId(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var model = domain.CreateModelContext(myCtx)
	var req GetUserBagReq
	if err := c.ShouldBindQuery(&req); err != nil {
		return myCtx, err
	}
	bag, err := bag_m.GetUserBag(model, req.BagId)
	if err != nil {
		return myCtx, err
	}
	gift, err := res_m.FindResGift(model, bag.ResId)
	if err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, user_cv.UserBag{
		BagId:      bag.ID,
		ResType:    bag.ResType,
		ResId:      bag.ResId,
		GiftId:     bag.ResId,
		Name:       gift.Name,
		DiamondNum: gift.DiamondNum,
		IconUrl:    gift.IconUrl,
		SvgaUrl:    gift.SvagUrl,
		Count:      bag.Count,
		RemainDays: int(bag.EndTime.Sub(time.Now()).Hours() / 24),
	})
	return myCtx, nil
}
