package cp_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/gin-gonic/gin"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/cv/cp_cv"
	"hilo-user/cv/user_cv"
	"hilo-user/domain/event/cp_ev"
	"hilo-user/domain/model/cp_m"
	"hilo-user/domain/model/user_m"
	"hilo-user/myerr/bizerr"
	"hilo-user/req"
	"hilo-user/resp"
	"time"
)

// @Tags CP v2
// @Summary cp空间
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param externalId query string true "查看用户的externalId"
// @Success 200 {object} cp_cv.CvSpace
// @Router /v2/cp/space [get]
func CpSpace(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, lang, err := req.GetUserIdLang(c, myContext)
	if err != nil {
		return myContext, err
	}
	externalId := c.Query("externalId")
	if len(externalId) <= 0 {
		return myContext, bizerr.InvalidParameter
	}
	var model = domain.CreateModelContext(myContext)
	userInfo, err := user_m.GetUser(model, userId)
	if err != nil {
		return myContext, err
	}
	targetUserInfo, err := user_m.GetUserByExtId(model, externalId)
	if err != nil {
		return myContext, err
	}
	var cpUserInfo *user_m.User
	expireAtUnix, cpLevel := int64(0), cp_e.CpLevel0
	visitTimes := int64(0)
	settlementDate, cpDays, applyToUnbind := "", 0, false
	nextPoints, curPoints := cp_e.CpLevelPoints[cp_e.CpLevel1], mysql.Num(0)
	cpRelation, exists := cp_m.GetCpRelation(model, targetUserInfo.ID)
	if exists {
		level := cp_m.GetCpLevel(model, cpRelation.ID)
		if level.ExpireAt.Before(time.Now()) {
			level.ExpireAt = time.Now().AddDate(0, 1, 0)
		}
		cpLevel, curPoints = level.Level, level.Points
		expireAtUnix, settlementDate = level.ExpireAt.Unix(), level.ExpireAt.Format("2006-01-02")
		cpDays = int(time.Now().Sub(cpRelation.CreatedTime).Hours()/24) + 1
		// cp 存在的时候,发布访问cp空间事件
		if err := cp_ev.PublishCpSpaceVisit(model, &cp_ev.SpaceVisitEvent{
			UserId:  userId,
			CpId:    cpRelation.ID,
			UserId1: cpRelation.UserId1,
			UserId2: cpRelation.UserId2,
		}); err != nil {
			return myContext, err
		}
		visitTimes = cp_m.CountCpSpaceVisitors(model, cpRelation.ID)
		cpUserId := cpRelation.UserId2
		if cpUserId == userId {
			cpUserId = cpRelation.UserId1
		}
		cpUserInfo, err = user_m.GetUser(model, cpUserId)
		if err != nil {
			return myContext, err
		}
		// 查看别人的cp空间
		if userId != cpRelation.UserId1 && userId != cpRelation.UserId2 {
			userInfo, err = user_m.GetUser(model, cpRelation.UserId1)
			if err != nil {
				return myContext, err
			}
		} else {
			// 自己的空间
			applyToUnbind = cp_m.GetApplyToUnbind(model, userId, cpUserId)
		}
	}
	if cpLevel != cp_e.CpLevelMax {
		nextPoints = cp_e.CpLevelPoints[cpLevel+1]
	} else {
		nextPoints = cp_e.CpLevelPoints[cp_e.CpLevelMax]
	}
	userPrivileges, err := cp_m.MGetUserSvipPrivilege(model, []uint64{userId})
	if err != nil {
		return myContext, err
	}
	allPrivilegeList := make([][]cp_cv.CvPrivilege, cp_e.CpLevelMax+1)
	for level := cp_e.CpLevel0; level <= cp_e.CpLevelMax; level++ {
		privilegeList := cp_cv.CopyCpLevelPrivilegeList(level, lang)
		for i, v := range privilegeList {
			if v.CanSwitch {
				privilegeList[i].UserSwitch = userPrivileges[userId][v.Type]
			}
		}
		allPrivilegeList[level] = privilegeList
	}
	// 返回值
	response := cp_cv.CvSpace{
		CpLevel: cp_cv.CvCpLevel{
			Level:          cpLevel,
			Points:         cp_e.CpLevelPoints[cpLevel] + curPoints,
			StartPoints:    cp_e.CpLevelPoints[cpLevel],
			EndPoints:      nextPoints,
			ExpireAtUnix:   expireAtUnix,
			SettlementDate: settlementDate,
			MaxLevel:       cp_e.CpLevelMax,
		},
		ResLevelList:  cp_cv.CvResLevelList,
		PrivilegeList: allPrivilegeList,
	}
	response.CpInfo = cp_cv.CvCpInfo{
		UserInfo:      user_cv.UserToTiny(*userInfo),
		CpDays:        cpDays,
		VisitTimes:    visitTimes,
		ApplyToUnbind: applyToUnbind,
	}
	if cpUserInfo != nil {
		response.CpInfo.CpUserInfo = user_cv.UserToTiny(*cpUserInfo)
	}
	// 需要提醒的纪念日
	if exists && (userId == cpRelation.UserId1 || userId == cpRelation.UserId2) {
		anniversary := cp_m.GetUserTodayCpAnniversary(model, cpRelation.ID)
		for _, a := range anniversary {
			response.CpAnniversary = append(response.CpAnniversary, cp_cv.CvCpAnniversary{
				Id:        a.ID,
				Content:   a.Content,
				Timestamp: a.Timestamp,
			})
		}
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

type CpPrivilegeOpenCloseReq struct {
	Type      cp_e.CpPrivilege `form:"type" binding:"required"`
	OpenClose mysql.OpenClose  `form:"openClose" binding:"required"`
}

// @Tags CP v2
// @Summary cp特权开关
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param type formData int true "1.空间 2.横幅 3.等级勋章 4.证书 5.进场特效 6.头像头饰 7.动态资料卡 8.麦位特效"
// @Param openClose formData int true "1:open 2:close"
// @Success 200
// @Router /v2/cp/privilege/openClose [put]
func CpPrivilegeOpenClose(c *gin.Context) (*mycontext.MyContext, error) {
	myCtx := mycontext.CreateMyContext(c.Keys)
	var request CpPrivilegeOpenCloseReq
	if err := c.ShouldBind(&request); err != nil {
		return myCtx, err
	}
	userId, err := req.GetUserId(c)
	if err != nil {
		return myCtx, err
	}
	var model = domain.CreateModelContext(myCtx)
	if err := cp_m.OpenCLoseUserSvipPrivilege(model, userId, request.Type, request.OpenClose); err != nil {
		return myCtx, err
	}
	resp.ResponseOk(c, "")
	return myCtx, nil
}
