package event_s

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/config"
	"git.hilo.cn/hilo-common/resource/redisCli"
	"github.com/jinzhu/now"
	"hilo-user/_const/enum/cp_e"
	"hilo-user/_const/enum/gift_e"
	"hilo-user/domain/event/gift_ev"
	"hilo-user/domain/model/cp_m"
	"time"
)

// 送礼增加cp等级
// 送礼增加cp排行榜
func CpGiftEvent() {
	gift_ev.AddSendGiftEventSync(func(model *domain.Model, event interface{}) error {
		sendGiftEvent, ok := event.(*gift_ev.SendGiftEvent)
		if !ok {
			model.Log.Errorf("AddSendGiftEventAsync event type err")
			return nil
		}
		// 只处理cp礼物
		if !sendGiftEvent.ResGift.Cp {
			return nil
		}
		for _, receiverUid := range sendGiftEvent.ReceiveUserIds {
			diamonds := sendGiftEvent.GiftN * sendGiftEvent.ResGift.DiamondNum
			// 有cp关系
			if cpRelation, exits := cp_m.GetCpRelationPair(model, sendGiftEvent.SendUserId, receiverUid); exits {
				if err := cp_m.AddCpLevelPoints(model, cpRelation, diamonds, sendGiftEvent.SceneType, sendGiftEvent.SceneUid); err != nil {
					model.Log.Errorf("AddCpLevelPoints fail:%v", err)
				}
				if err := cp_m.AddCpDayRank(model, cpRelation, diamonds); err != nil {
					model.Log.Errorf("AddCpDayRank fail:%v", err)
				}
				// 检查最新的等级
				if cpLevel := cp_m.GetCpLevel(model, cpRelation.Id); cpLevel.CpId >= 0 {
					points := cpLevel.Points + cp_e.CpLevelPoints[cpLevel.Level]
					if err := cp_m.UpdateCpAchievement(model, cpLevel.CpId, cpRelation.UserId1, cpRelation.UserId2, cp_e.CpAchievementLevel, points); err != nil {
						model.Log.Errorf("UpdateCpAchievement fail:%v", err)
					}
				}
				// 检查最高的分数
				for _, queryType := range []string{"day", "week", "month"} {
					var beginDate, endDate string
					var cpAchievementType cp_e.CpAchievement
					switch queryType {
					case "day":
						beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
						cpAchievementType = cp_e.CpAchievementDayRank
					case "week":
						beginDate = now.BeginningOfWeek().Format("2006-01-02")
						endDate = now.EndOfWeek().Format("2006-01-02")
						cpAchievementType = cp_e.CpAchievementWeekRank
					case "month":
						beginDate = now.BeginningOfMonth().Format("2006-01-02")
						endDate = now.EndOfMonth().Format("2006-01-02")
						cpAchievementType = cp_e.CpAchievementMonthRank
					}
					if data := cp_m.GetCpDayRank(model, beginDate, endDate, cpRelation.Id); data.Score > 0 {
						if err := cp_m.UpdateCpAchievement(model, cpRelation.Id, cpRelation.UserId1, cpRelation.UserId2, cpAchievementType, data.Score); err != nil {
							model.Log.Errorf("UpdateCpAchievement fail:%v", err)
						}
					}
				}
				// 检查最新日周月榜单
				return nil // 业务场景允许提前break(cp是唯一的)
			}
		}
		return nil
	})
}

// 漏掉加分数的送礼记录，补上cp增加分数
// 送礼增加cp等级
// 送礼增加cp排行榜
func CompensateCpScore() {
	if !config.IsMaster() {
		return
	}
	model := domain.CreateModelNil()
	num, err := redisCli.IncrNumExpire("scirpt:compensateCpScore", 1, time.Hour*999)
	if err != nil {
		model.Log.Errorf("CompensateCpScore err:%v", err)
	}
	if num > 1 {
		model.Log.Errorf("CompensateCpScore 执行过了，退出")
		return
	}
	// 查找需要补分的送礼记录
	type GiftOperate struct {
		Id              uint64    `json:"id"`
		ResGiftId       uint64    `json:"res_gift_id"`
		GiftN           int32     `json:"gift_n"`
		SendUserId      uint64    `json:"send_user_id"`
		ReceiveUserId   uint64    `json:"receive_user_id"`
		SendUserDiamond uint32    `json:"send_user_diamond"`
		SceneType       int8      `json:"scene_type"`
		SceneUid        string    `json:"scene_uid"`
		CreatedTime     time.Time `json:"created_time"`
	}
	rows := make([]*GiftOperate, 0)
	err = model.DB().Model(GiftOperate{}).
		Where("created_time < ?", "2023-06-22 11:20:15").
		Where("res_gift_id in (?)",
			[]int{3561, 3571, 3581, 3591, 3601, 3611, 3621, 3631, 3641, 3651, 3661, 3671, 3681, 3691, 3701, 3711}).
		Find(&rows).Error
	if err != nil {
		model.Log.Errorf("CompensateCpScore err:%v", err)
		return
	}
	model.Log.Infof("CompensateCpScore len(rows):%v", len(rows))

	for idx, r := range rows {
		diamonds := r.SendUserDiamond
		// 有cp关系
		if cpRelation, exits := cp_m.GetCpRelationPair(model, r.SendUserId, r.ReceiveUserId); exits {
			if err := cp_m.AddCpLevelPoints(model, cpRelation, diamonds, gift_e.GiftOperateSceneType(r.SceneType), r.SceneUid); err != nil {
				model.Log.Errorf("CompensateCpScore AddCpLevelPoints fail:%v", err)
				return
			}
			if err := cp_m.AddCpDayRank(model, cpRelation, diamonds); err != nil {
				model.Log.Errorf("CompensateCpScore AddCpDayRank fail:%v", err)
				return
			}
			// 检查最新的等级
			if cpLevel := cp_m.GetCpLevel(model, cpRelation.Id); cpLevel.CpId >= 0 {
				points := cpLevel.Points + cp_e.CpLevelPoints[cpLevel.Level]
				if err := cp_m.UpdateCpAchievement(model, cpLevel.CpId, cpRelation.UserId1, cpRelation.UserId2, cp_e.CpAchievementLevel, points); err != nil {
					model.Log.Errorf("CompensateCpScore UpdateCpAchievement fail:%v", err)
				}
			}
			// 检查最高的分数
			for _, queryType := range []string{"day", "week", "month"} {
				var beginDate, endDate string
				var cpAchievementType cp_e.CpAchievement
				switch queryType {
				case "day":
					beginDate, endDate = r.CreatedTime.Format("2006-01-02"), r.CreatedTime.Format("2006-01-02")
					cpAchievementType = cp_e.CpAchievementDayRank
				case "week":
					beginDate = now.With(r.CreatedTime).BeginningOfWeek().Format("2006-01-02")
					endDate = now.With(r.CreatedTime).EndOfWeek().Format("2006-01-02")
					cpAchievementType = cp_e.CpAchievementWeekRank
				case "month":
					beginDate = now.BeginningOfMonth().Format("2006-01-02")
					endDate = now.EndOfMonth().Format("2006-01-02")
					cpAchievementType = cp_e.CpAchievementMonthRank
				}
				if data := cp_m.GetCpDayRank(model, beginDate, endDate, cpRelation.Id); data.Score > 0 {
					if err := cp_m.UpdateCpAchievement(model, cpRelation.Id, cpRelation.UserId1, cpRelation.UserId2, cpAchievementType, data.Score); err != nil {
						model.Log.Errorf("CompensateCpScore UpdateCpAchievement fail:%v", err)
					}
				}
			}
		}
		if idx%1000 == 0 {
			time.Sleep(time.Millisecond * 100)
		}
	}

	model.Log.Infof("CompensateCpScore 补偿完毕")
}
