package main

import (
	"fmt"
	"git.hilo.cn/hilo-common/script/model"
	"git.hilo.cn/hilo-common/script/mysql"
	"time"
)

type FruitChargeData struct {
	Date             string
	Country          string
	UserId           uint64
	Code             string
	ChargeMoney      float64
	FirstChargeDate  string
	ChargeDiamond    int64
	FruitDiamond     int64
	FruitAwardPStake int64
	GiftDiamond      int64
}

var beginTime = "2023-02-13 00:00:00"
var endTime = "2023-02-14 00:00:00"
var uCodes = []string{"77707", "1010", "2575", "9160", "3022", "4050", "1412428", "85085", "24302", "9163", "443443", "1231", "28828", "31333", "67889", "77888", "1501402", "1467912", "12302", "19999", "8687", "70033", "50555", "33533", "10002", "15302", "1515761", "99777", "13133", "1412", "313333", "334433", "10100", "288255"}

func main() {
	var data []FruitChargeData
	for _, code := range uCodes {
		var tmp FruitChargeData
		u, _ := GetUserByCode(code)
		tmp.Code, tmp.Country = u.Code, u.Country
		chargeMoney, chargeDiamond, _ := GetChargeMoneyDiamond(u.Id)
		tmp.ChargeMoney = float64(chargeMoney)
		tmp.ChargeDiamond = chargeDiamond
		tmp.FirstChargeDate, _ = GetFirstChargeDate(u.Id)
		tmp.FruitDiamond, _ = GetDiamondSumOperateType(u.Id, 39)
		award, _ := GetDiamondSumOperateType(u.Id, 38)
		tmp.FruitAwardPStake = award - tmp.FruitDiamond
		tmp.GiftDiamond, _ = GetDiamondSumOperateType(u.Id, 1)
		data = append(data, tmp)
	}
	for _, d := range data {
		fmt.Printf("%v|%v|%v|%v|%v|%v|%v\n", d.Code, d.ChargeMoney/100, d.FirstChargeDate, d.ChargeDiamond, d.FruitDiamond, d.FruitAwardPStake, d.GiftDiamond)
	}
}

func GetUserByCode(code string) (model.User, error) {
	var u model.User
	if err := mysql.ProdReadOnlyDB.Model(model.User{}).Where("code = ?", code).First(&u).Error; err != nil {
		panic(err)
	}
	return u, nil
}

func GetChargeMoneyDiamond(uid uint64) (int64, int64, error) {
	type R struct {
		Money   int64
		Diamond int64
	}
	var money R
	if err := mysql.ProdReadOnlyDB.Table("diamond_account_detail AS d").
		Joins("INNER JOIN pay_order AS p ON d.origin_id = p.id").
		Where("d.user_id in (?) AND operate_type in (?) AND add_reduce = 1",
			uid, []int{int(4), int(42), int(55), int(68)}).
		Where("d.created_time >= ? AND d.created_time < ?", beginTime, endTime).
		Order("d.user_id").
		Select("SUM(p.price) as money,SUM(d.num) as diamond").Scan(&money).Error; err != nil {
		return 0, 0, err
	}
	type R2 struct {
		Price   int64
		Diamond int64
	}
	var money2 R2
	if err := mysql.ProdReadOnlyDB.Table("diamond_account_detail AS d").
		Joins("INNER JOIN dealer_transfer_detail AS t ON d.origin_id = t.id").
		Where("d.user_id in (?) AND operate_type = ? AND add_reduce = 1",
			uid, 26).Select("SUM(t.dollar) as price,SUM(d.num) as diamond").
		Where("d.created_time >= ? AND d.created_time < ?", beginTime, endTime).
		Order("d.user_id").
		Scan(&money2).Error; err != nil {
		return 0, 0, err
	}
	return money.Money + money2.Price, money.Diamond + money2.Diamond, nil
}

func GetFirstChargeDate(uid uint64) (string, error) {
	var t time.Time
	if err := mysql.ProdReadOnlyDB.Table("diamond_account_detail AS d").
		Joins("INNER JOIN pay_order AS p ON d.origin_id = p.id").
		Where("d.user_id in (?) AND operate_type in (?) AND add_reduce = 1",
			uid, []int{int(4), int(42), int(55), int(68)}).
		Order("d.created_time").
		Limit(1).
		Select("d.created_time").Scan(&t).Error; err != nil {
		return "", err
	}
	var t2 time.Time
	if err := mysql.ProdReadOnlyDB.Table("diamond_account_detail AS d").
		Joins("INNER JOIN dealer_transfer_detail AS t ON d.origin_id = t.id").
		Where("d.user_id in (?) AND operate_type = ? AND add_reduce = 1", uid, 26).
		Order("d.created_time").
		Limit(1).
		Select("d.created_time").Scan(&t2).Error; err != nil {
		return "", err
	}
	if t.IsZero() {
		t = t2
	}
	if !t2.IsZero() && t2.Before(t) {
		t = t2
	}
	return t.Format("2006-01-02"), nil
}

// operType: 1:送礼 38:水果机奖励 39:水果机投注
func GetDiamondSumOperateType(uid uint64, operType int) (int64, error) {
	var num int64
	if err := mysql.ProdReadOnlyDB.Table("diamond_account_detail").
		Where("user_id = ? AND operate_type = ?", uid, operType).
		Where("created_time >= ? AND created_time < ?", beginTime, endTime).
		Group("user_id").
		Select("SUM(num) as num").
		Scan(&num).Error; err != nil {
		return num, err
	}
	return num, nil
}
