package rpc

import (
	"encoding/json"
	"fmt"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mylogrus"
	"git.hilo.cn/hilo-common/resource/consul"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/hashicorp/consul/api"
	"math/rand"
)

const (
	defaultUserConsulName   = "hiloUser"
	defaultUserServerScheme = "http"
	defaultUserServerAddr   = "127.0.0.1:9040" // 默认内网转发,本地回环
)

var UserServerHost = []string{defaultUserServerAddr}

func init() {
	go func() {
		address := api.DefaultConfig().Address // 用consul api的default config
		if err := consul.RegisterWatcher("services", nil, address, func(serviceStatus map[string]map[string][]string) {
			if statusAddrs, ok := serviceStatus[defaultUserConsulName]; ok {
				healthAddrs, _ := statusAddrs[api.HealthPassing]
				l := len(healthAddrs)
				if l > 0 {
					mylogrus.MyLog.Infof("consul service update state:%v-%v", defaultUserConsulName, healthAddrs)
					UserServerHost = healthAddrs
				} else {
					mylogrus.MyLog.Warnf("consul service update local state:%v-%v", defaultUserConsulName, defaultUserServerAddr)
					UserServerHost = []string{defaultUserServerAddr} // 有其他问题都用默认的
				}
				for status := range statusAddrs {
					if status == api.HealthPassing {
						continue
					}
					mylogrus.MyLog.Warnf("consul service wrong state:%v-%v-%v", defaultUserConsulName, status, statusAddrs[status])
				}
			}
		}); err != nil {
			mylogrus.MyLog.Errorf("启动 consul 的watch监控失败")
		}
	}()
}

type CvUserLevel struct {
	UserId          mysql.ID `json:"userId"`          // 用户id
	WealthUserGrade uint32   `json:"wealthUserGrade"` //财富等级
	CharmUserGrade  uint32   `json:"charmUserGrade"`  //魅力等级
}

// 获取用户等级
func GetUserLevel(model *domain.Model, userId mysql.ID) (CvUserLevel, error) {
	level, err := MGetUserLevel(model, []mysql.ID{userId})
	if err != nil {
		return CvUserLevel{}, nil
	}
	return level[userId], nil
}

// 批量获取用户等级
func MGetUserLevel(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]CvUserLevel, error) {
	type Response struct {
		Code    int                      `json:"code"`
		Message string                   `json:"message"`
		Data    map[mysql.ID]CvUserLevel `json:"data"`
	}
	var res = make(map[mysql.ID]CvUserLevel)
	if len(userIds) <= 0 {
		return res, nil
	}
	var userIdsStr []string
	for _, userId := range userIds {
		userIdsStr = append(userIdsStr, fmt.Sprintf("%d", userId))
	}
	_url := fmt.Sprintf("%v://%v/inner/user/levels", defaultUserServerScheme, getUserHost())
	resp, err := HttpGet(model, _url, nil, map[string][]string{
		"ids": userIdsStr,
	})
	if err != nil {
		model.Log.Errorf("MGetUserLevel fail:%v", err)
		return res, err
	}
	response := new(Response)
	if err = json.Unmarshal(resp, response); err != nil {
		model.Log.Errorf("MGetUserLevel json fail:%v", err)
		return res, err
	}
	for userId, level := range response.Data {
		res[userId] = CvUserLevel{
			UserId:          level.UserId,
			WealthUserGrade: level.WealthUserGrade,
			CharmUserGrade:  level.CharmUserGrade,
		}
	}
	return res, nil
}

type CvUserBag struct {
	BagId      mysql.ID   `json:"bagId"`      // 背包id
	ResType    mysql.Type `json:"resType"`    // 道具类型 1:礼物道具
	ResId      mysql.ID   `json:"resId"`      // 道具资源id
	Name       string     `json:"name"`       // 资源名称
	DiamondNum mysql.Num  `json:"diamondNum"` // 钻石数量
	IconUrl    string     `json:"iconUrl"`    // icon url
	SvgaUrl    string     `json:"svgaUrl"`    // svga url
	Count      mysql.Num  `json:"count"`      // 拥有数量
	RemainDays int        `json:"remainDays"` // 有效天数
}

// 根据bagId获取用户背包
func GetUserBagId(model *domain.Model, bagId mysql.ID) (CvUserBag, error) {
	type Response struct {
		Code    int       `json:"code"`
		Message string    `json:"message"`
		Data    CvUserBag `json:"data"`
	}
	_url := fmt.Sprintf("%v://%v/inner/user/bag/id", defaultUserServerScheme, getUserHost())
	resp, err := HttpGet(model, _url, nil, map[string][]string{
		"bagId": {fmt.Sprintf("%d", bagId)},
	})
	if err != nil {
		model.Log.Errorf("GetUserBag fail:%v", err)
		return CvUserBag{}, err
	}
	response := new(Response)
	if err = json.Unmarshal(resp, response); err != nil {
		model.Log.Errorf("GetUserBag json fail:%v", err)
		return CvUserBag{}, err
	}
	return response.Data, nil
}

func getUserHost() string {
	l := len(UserServerHost)
	r := rand.Intn(l) // 随机一个
	mylogrus.MyLog.Infof("getHostUser:%v---%v", r, UserServerHost[r])
	return UserServerHost[r]
}
