package aws

import (
	"context"
	"errors"
	"git.hilo.cn/hilo-common/mylogrus"
	hiloCfg "git.hilo.cn/hilo-common/resource/config"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/aws/aws-sdk-go-v2/aws"
	"github.com/aws/aws-sdk-go-v2/config"
	"github.com/aws/aws-sdk-go-v2/service/rekognition"
	"github.com/aws/aws-sdk-go-v2/service/rekognition/types"
	"github.com/aws/aws-sdk-go-v2/service/s3"
	uuid "github.com/satori/go.uuid"
	"github.com/sirupsen/logrus"
	"log"
	"mime/multipart"
	"net/http"
	"strings"
)

var cfg aws.Config

func init() {
	// Load the Shared AWS Configuration (~/.aws/config)
	var err error = nil
	cfg, err = config.LoadDefaultConfig(context.TODO())
	if err != nil {
		log.Fatal(err)
	}
	mylogrus.MyLog.Infof("aws init done: %+v\n", cfg)
}

func GetUUID() string {
	return strings.Replace(uuid.NewV4().String(), "-", "", -1)
}

func UploadImage(log *logrus.Entry, imagPath string) (string, error) {
	resp, _ := http.Get(imagPath)

	// Create an Amazon S3 service client
	client := s3.NewFromConfig(cfg)

	fileName := hiloCfg.GetConfigAws().AWS_DIR + "image/" + GetUUID()
	ct := "image/jpeg"
	response, err := client.PutObject(context.TODO(), &s3.PutObjectInput{
		Bucket:        aws.String(hiloCfg.GetConfigAws().AWS_BUCKET),
		Key:           aws.String(fileName),
		Body:          resp.Body,
		ContentLength: resp.ContentLength,
		ContentType:   &ct,
	})
	log.Infof("UploadImage, err: %+v rsp: %+v", err, response)

	if err != nil {
		return "", err
	}

	return fileName, nil
}

func UploadImageFile(log *logrus.Entry, imagFile multipart.File, t string, contentType *string) (string, error) {
	// Create an Amazon S3 service client
	client := s3.NewFromConfig(cfg)

	log.Infof("UploadImageFile, config: %+v", cfg)
	fileName := hiloCfg.GetConfigAws().AWS_DIR + "manager/" + t
	response, err := client.PutObject(context.TODO(), &s3.PutObjectInput{
		Bucket:      aws.String(hiloCfg.GetConfigAws().AWS_BUCKET),
		Key:         aws.String(fileName),
		Body:        imagFile,
		ContentType: contentType,
	})
	log.Infof("UploadImageFile, err: %+v, rsp: %+v", err, response)

	if err != nil {
		return "", err
	}
	return fileName, nil
}

func ModerateLabels(logger *logrus.Entry, userId uint64, fileName string) (bool, error) {
	logger.Infof("user %d, fileName: %s", userId, fileName)

	client := rekognition.NewFromConfig(cfg)
	confidence := hiloCfg.GetConfigAws().CONFIDENCE
	output, err := client.DetectModerationLabels(context.TODO(), &rekognition.DetectModerationLabelsInput{
		Image: &types.Image{
			S3Object: &types.S3Object{
				Bucket: aws.String(hiloCfg.GetConfigAws().AWS_BUCKET),
				Name:   aws.String(fileName),
			},
		},
		MinConfidence: &confidence,
	})
	if err != nil {
		return false, err
	}
	if output == nil {
		return false, errors.New("IncorrectState")
	}
	logger.Infof("user %d, file %s, detect labels: %+v", userId, fileName, output.ModerationLabels)
	labels := ""
	for _, i := range output.ModerationLabels {
		if i.Name != nil {
			labels = labels + *i.Name + " "
		}
	}
	mr := ModerateRecord{UserId: userId, FileName: fileName, Labels: labels}
	if err = mysql.Db.Create(&mr).Error; err != nil {
		logger.Infof("Save ModerateRecord err: %v", err)
	}
	return len(output.ModerationLabels) <= 0, nil
}
