package rpc

import (
	"encoding/json"
	"fmt"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mylogrus"
	"git.hilo.cn/hilo-common/resource/consul"
	"git.hilo.cn/hilo-common/resource/mysql"
	"math/rand"
)

const (
	defaultFinanceConsulName   = "hiloFinance"
	defaultFinanceServerScheme = "http"
	defaultFinanceServerAddr   = "127.0.0.1:9030" // 默认内网转发,本地回环
)

var financeServerHost = []string{defaultFinanceServerAddr}

func init() {
	go func() {
		consul.RegisterWatcher(defaultFinanceConsulName, func(addr []string) {
			if len(addr) > 0 {
				financeServerHost = addr
			}
		})
	}()
}

type CvSvip struct {
	SvipLevel  int               `json:"svipLevel"`
	Privileges []CVSvipPrivilege `json:"privileges"` // 特权
}

type CVSvipPrivilege struct {
	Type        int    `json:"type"`                  // 1.专属勋章, 2.专属标识, 3.设备和IP踢出房间, 4.隐藏在线, 5.禁止跟随, 6.炫彩昵称, 7.隐藏礼物墙, 8.隐藏访客记录, 9.排行榜隐身, 10.房间防踢, 11.房间防抱下麦
	CanSwitch   bool   `json:"canSwitch"`             // 能否开关
	UserSwitch  bool   `json:"userSwitch"`            // 用户开关
	MysteryCode string `json:"mysteryCode,omitempty"` // 神秘人代码,特权17的专属代码
}

// 获取svip等级
func GetUserSvip(model *domain.Model, userId mysql.ID) (CvSvip, error) {
	svip, err := MGetUserSvip(model, []mysql.ID{userId})
	if err != nil {
		return CvSvip{}, nil
	}
	return svip[userId], nil
}

// 批量获取svip等级
// map userId->level
func MGetUserSvipLevel(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]int, error) {
	res := make(map[mysql.ID]int, len(userIds))
	if len(userIds) <= 0 {
		return res, nil
	}
	level, err := MGetUserSvip(model, userIds)
	if err != nil {
		return res, nil
	}
	for userId, svip := range level {
		res[userId] = svip.SvipLevel
	}
	return res, nil
}

// 批量获取svip
func MGetUserSvip(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]CvSvip, error) {
	type Response struct {
		Code    int    `json:"code"`
		Message string `json:"message"`
		Data    map[mysql.ID]CvSvip
	}
	var res = make(map[mysql.ID]CvSvip)
	if len(userIds) <= 0 {
		return res, nil
	}
	var userIdsStr []string
	for _, userId := range userIds {
		userIdsStr = append(userIdsStr, fmt.Sprintf("%d", userId))
	}
	_url := fmt.Sprintf("%v://%v/inner/svip/users", defaultFinanceServerScheme, getFinanceHost())
	resp, err := HttpGet(model, _url, nil, map[string][]string{
		"ids": userIdsStr,
	})
	if err != nil {
		model.Log.Errorf("MGetUserSvip fail:%v", err)
		return res, err
	}
	response := new(Response)
	if err = json.Unmarshal(resp, response); err != nil {
		model.Log.Errorf("MGetUserSvip json fail:%v", err)
		return res, err
	}
	for userId, svip := range response.Data {
		res[userId] = CvSvip{
			SvipLevel:  svip.SvipLevel,
			Privileges: response.Data[userId].Privileges,
		}
	}
	return res, nil
}

// 拷贝客户端用到的privileges赋值
// 避免推送大包体
// 目前暂时只需要 svip7.神秘人特权
func CopySimpleSvip(svip CvSvip) CvSvip {
	var simpleSvip CvSvip
	simpleSvip.SvipLevel = svip.SvipLevel
	for _, p := range svip.Privileges {
		if p.Type == 17 { // 神秘人代码,特权17的专属代码
			simpleSvip.Privileges = append(simpleSvip.Privileges, CVSvipPrivilege{
				Type:        p.Type,
				CanSwitch:   p.CanSwitch,
				UserSwitch:  p.UserSwitch,
				MysteryCode: p.MysteryCode,
			})
		}
	}
	return simpleSvip
}

func getFinanceHost() string {
	l := len(financeServerHost)
	r := rand.Intn(l) // 随机一个
	mylogrus.MyLog.Infof("getHostFinance:%v---%v", r, financeServerHost[r])
	return financeServerHost[r]
}
