package rpc

import (
	"encoding/json"
	"git.hilo.cn/hilo-common/mylogrus"
	"git.hilo.cn/hilo-common/protocol/userProxy"
	"git.hilo.cn/hilo-common/utils"
	"google.golang.org/protobuf/proto"
)

func SendFruitMachine(date string, round uint32) error {
	msg := &userProxy.FruitMachine{
		Date:  date,
		Round: round,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := broadcast(MsgFruitMachine, buffer)

		//记录socket,注意闭包问题
		go func(userId uint64, msg *userProxy.FruitMachine, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgFruitMachine, userId, string(buf[:]), rspUids, err)
		}(0, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendFruitMachine send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendFruitMachine send success")
		}
	} else {
		return err
	}
	return nil
}

func SendGlobalRocketNotice(groupId string, period string, round uint32, stage uint32, fromUserId uint64, topUserIcon string, nick string, code string, avatar string) error {
	msg := &userProxy.GlobalRocketNotice{
		GroupId:     groupId,
		Period:      period,
		Round:       round,
		Stage:       stage,
		TopUserIcon: topUserIcon,
		Nick:        nick,
		Code:        code,
		Avatar:      avatar,
	}

	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := broadcast(MsgTypeGlobalRocketNotice, buffer)

		//记录socket,注意闭包问题
		go func(userId uint64, msg *userProxy.GlobalRocketNotice, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgTypeGlobalRocketNotice, userId, string(buf[:]), rspUids, err)
		}(fromUserId, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc GlobalRocketNotice send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc GlobalRocketNotice send success")
		}
	} else {
		return err
	}
	return nil
}

// 发送钻石变化通知
func SendDiamondChange(userId uint64, diamond, pinkDiamond uint32) error {
	msg := &userProxy.DiamondChange{
		RemainDiamond:     diamond,
		RemainPinkDiamond: pinkDiamond,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := multicast([]uint64{userId}, MsgDiamondChange, buffer)
		//记录socket,注意闭包问题
		go func(userId uint64, msg *userProxy.DiamondChange, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgDiamondChange, userId, string(buf[:]), rspUids, err)
		}(userId, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendDiamondChange send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendDiamondChange send success")
		}
	} else {
		return err
	}
	return nil
}

// 发送游戏横幅
// param winUserId:胜利的用户id
// param gameType 0:slot 5:luckybox 6:fruit
// param gameId 7:slot 8:candy 5:luckybox 6:fruit
func SendGlobalGameBanner(winUserId uint64, diamond uint64, avatar string, gameId uint64, gameType uint32) error {
	bannerUrl := ""
	switch gameType {
	case 0:
		bannerUrl = "https://image.whoisamy.shop/hilo/resource/game/game_banner_slot.png"
	case 5:
		bannerUrl = "https://image.whoisamy.shop/hilo/resource/game/game_banner_luckybox.png"
	case 6:
		bannerUrl = "https://image.whoisamy.shop/hilo/resource/game/game_banner_fruit.png"
	}
	if gameId == uint64(8) {
		bannerUrl = "https://image.whoisamy.shop/hilo/resource/game/game_banner_candy.png"
	}
	msg := &userProxy.GlobalGameBanner{
		GameId:    gameId,
		GameType:  gameType,
		UserId:    winUserId,
		Avatar:    avatar,
		Diamond:   diamond,
		BannerUrl: bannerUrl,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := broadcast(MsgTypeGlobalGameBanner, buffer)

		//记录socket,注意闭包问题
		go func(userId uint64, msg *userProxy.GlobalGameBanner, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgTypeGlobalGameBanner, userId, string(buf[:]), rspUids, err)
		}(winUserId, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendGlobalGameBanner send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendGlobalGameBanner send success")
		}
	} else {
		return err
	}
	return nil
}

func SendJoinGroup(userId uint64, externalId string, groupId string) error {
	msg := &userProxy.JoinGroup{GroupId: groupId, ExternalId: externalId}
	if buffer, err := proto.Marshal(msg); err == nil {
		userIds := []uint64{userId}
		rspUids, err := multicast(userIds, MsgTypeJoinGroup, buffer)

		//记录socket,注意闭包问题
		go func(userIds []uint64, msg *userProxy.JoinGroup, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLogs(MsgTypeJoinGroup, userIds, string(buf[:]), rspUids, err)
		}(userIds, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendJoinGroup send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendJoinGroup send success")
		}
	} else {
		return err
	}
	return nil
}

func SendConfigChange(operUserId uint64, configType uint32) error {
	msg := &userProxy.ConfigChange{
		Type: configType,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := broadcast(MsgTypeConfigChange, buffer)

		//记录socket,注意闭包问题
		go func(configType uint32, msg *userProxy.ConfigChange, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgTypeConfigChange, operUserId, string(buf[:]), rspUids, err)
		}(configType, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendConfigChange send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendConfigChange send success")
		}
	} else {
		return err
	}
	return nil
}

func SendGroupChatNotice(fromUserId uint64, userIds []uint64, senderExtId string, senderCode string, senderSex uint32, senderAvatar string,
	text string, groupId, groupName, groupCode, groupAvatar string, userInNum uint32) error {
	msg := &userProxy.GroupSendNotice{
		SenderExtId:  senderExtId,
		SenderCode:   senderCode,
		SenderSex:    senderSex,
		SenderAvatar: senderAvatar,
		Text:         text,
		GroupName:    groupName,
		GroupCode:    groupCode,
		GroupAvatar:  groupAvatar,
		UserInNum:    userInNum,
		GroupId:      groupId,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		rspUids, err := multicast(userIds, MsgTypeGroupChatNotice, buffer)

		//记录socket,注意闭包问题
		go func(userId uint64, msg *userProxy.GroupSendNotice, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLog(MsgTypeGroupChatNotice, userId, string(buf[:]), rspUids, err)
		}(fromUserId, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendGroupChatNotice send fail,userId:%v", fromUserId)
			return err
		} else {
			mylogrus.MyLog.Infof("grpc SendGroupChatNotice send success,userId:%v", fromUserId)
		}
	} else {
		return err
	}
	return nil
}

// 羊羊匹配成功
func SendSheepMatchSuccess(matchId, myUserId, otherUserId uint64, nick1, nick2, avatar1, avatar2 string) error {
	msg := &userProxy.SheepMatchSuccess{
		MatchId:   matchId,
		User:      &userProxy.User{Id: myUserId, Nick: nick1, Avatar: avatar1},
		OtherUser: &userProxy.User{Id: otherUserId, Nick: nick2, Avatar: avatar2},
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		userIds := []uint64{myUserId}
		rspUids, err := multicast(userIds, MsgTypeSheepGameMatchSuccess, buffer)

		//记录socket,注意闭包问题
		go func(userIds []uint64, msg *userProxy.SheepMatchSuccess, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLogs(MsgTypeSheepGameMatchSuccess, userIds, string(buf[:]), rspUids, err)
		}(userIds, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendSheepMatchSuccess send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendSheepMatchSuccess send success")
		}
	} else {
		return err
	}
	return nil
}

// 羊羊游戏结果
func SendSheepGameResult(matchId, winId, userId1, userId2 uint64, nick1, nick2, avatar1, avatar2 string, isAiMap map[uint64]bool) error {
	if isAiMap == nil {
		isAiMap = make(map[uint64]bool)
	}
	var players []*userProxy.SheepGamePlayer
	players = append(players, &userProxy.SheepGamePlayer{IsWin: winId == userId1, User: &userProxy.User{
		Id: userId1, Nick: nick1, Avatar: avatar1,
	}})
	players = append(players, &userProxy.SheepGamePlayer{IsWin: winId == userId2, User: &userProxy.User{
		Id: userId2, Nick: nick2, Avatar: avatar2,
	}})
	if !players[0].IsWin {
		players[0], players[1] = players[1], players[0]
	}
	for i := range players {
		players[i].Rank = int32(i) + 1
	}
	msg := &userProxy.SheepGameResult{
		MatchId: matchId,
		Players: players,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		var userIds []uint64
		if !isAiMap[userId1] {
			userIds = append(userIds, userId1)
		}
		if !isAiMap[userId2] {
			userIds = append(userIds, userId2)
		}
		rspUids, err := multicast(userIds, MsgTypeSheepGameResult, buffer)

		//记录socket,注意闭包问题
		go func(userIds []uint64, msg *userProxy.SheepGameResult, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLogs(MsgTypeSheepGameResult, userIds, string(buf[:]), rspUids, err)
		}(userIds, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendSheepGameResult send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendSheepGameResult send success")
		}
	} else {
		return err
	}
	return nil
}

// cp邀请
func SendCpInviteNotice(userId uint64, code, name, avatar, content, extId string) error {
	defer utils.CheckGoPanic()
	msg := &userProxy.CpInvite{
		User: &userProxy.User{Id: userId, Code: code, Nick: name, Avatar: avatar, ExternalId: extId},
		Msg:  content,
	}
	if buffer, err := proto.Marshal(msg); err == nil {
		userIds := []uint64{userId}
		rspUids, err := multicast(userIds, MsgTypeCpInvite, buffer)

		//记录socket,注意闭包问题
		go func(userIds []uint64, msg *userProxy.CpInvite, rspUids []uint64, err error) {
			buf, _ := json.Marshal(msg)
			AddRpcLogs(MsgTypeCpInvite, userIds, string(buf[:]), rspUids, err)
		}(userIds, msg, rspUids, err)

		if err != nil {
			mylogrus.MyLog.Errorf("grpc SendCpInviteNotice send fail")
			return err
		} else {
			mylogrus.MyLog.Info("grpc SendCpInviteNotice send success")
		}
	} else {
		return err
	}
	return nil
}
