package group_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"gorm.io/gorm"
	"hilo-group/_const/enum/group_e"
	"hilo-group/_const/enum/res_e"
	"hilo-group/domain/model/res_m"
	"hilo-group/myerr"
	"time"
)

type GroupSupportResult struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	GroupUid      mysql.Str
	Grade         res_e.ResGroupSupportGrade
	Period        mysql.Str
	Heat          uint // 群热度
}

// 根据指定时间计算支持度时间范围：时间到上一个周一的沙特18PM，即北京时间23PM 到 下一个周一同样时间
func GetSupportLevelTime(t time.Time) (time.Time, time.Time, string) {
	t = utils.GetMonday(t)
	beginTime := time.Date(t.Year(), t.Month(), t.Day(), group_e.SUPPORT_LEVEL_BOUNDARY_HOUR, 0, 0, 0, time.Local)
	endTime := beginTime.AddDate(0, 0, group_e.SUPPORT_LEVEL_PERIOD_DAY)
	return beginTime, endTime, endTime.Format(utils.DATE_FORMAT)
}

func InitGroupSupportResult(model *domain.Model, groupUid string, grade res_e.ResGroupSupportGrade, period string, heat uint) *GroupSupportResult {
	return &GroupSupportResult{
		Model:    model,
		GroupUid: groupUid,
		Grade:    grade,
		Period:   period,
		Heat:     heat,
	}
}

func GetAllGroupSupportResult(db *gorm.DB, period string) (map[string]uint8, error) {
	if len(period) <= 0 {
		return nil, nil
	}
	gsr := GroupSupportResult{
		Period: period,
	}
	rows := make([]GroupSupportResult, 0)
	if err := db.Where(gsr).Find(&rows).Error; err != nil {
		return nil, err
	}
	result := make(map[string]uint8)
	for _, i := range rows {
		result[i.GroupUid] = i.Grade
	}
	return result, nil
}

//群组扶持奖励，利益分配者
type GroupSupportAwardAdmin struct {
	mysql.Entity
	*domain.Model     `gorm:"-"`
	GroupUid          mysql.Str
	IssuerUserId      mysql.ID
	UserId            mysql.ID
	DiamondNum        mysql.Num
	Grade             res_e.ResGroupSupportGrade
	ResGroupSupportId mysql.ID
	Period            mysql.Str
}

func (gsa *GroupSupportAwardAdmin) Get(db *gorm.DB) ([]GroupSupportAwardAdmin, error) {
	result := make([]GroupSupportAwardAdmin, 0)
	err := db.Model(&GroupSupportAwardAdmin{}).Where("group_uid = ? AND period >= ?", gsa.GroupUid, gsa.Period).Find(&result).Error
	if err != nil {
		return nil, err
	}
	return result, nil
}

//群组扶持奖励，助手
type GroupSupportAwardMgr struct {
	mysql.Entity
	*domain.Model     `gorm:"-"`
	GroupUid          mysql.Str
	IssuerUserId      mysql.ID
	UserId            mysql.ID
	DiamondNum        mysql.Num
	Grade             res_e.ResGroupSupportGrade
	ResGroupSupportId mysql.ID
	Period            mysql.Str
}

func (gsa *GroupSupportAwardMgr) Get(db *gorm.DB) ([]GroupSupportAwardMgr, error) {
	result := make([]GroupSupportAwardMgr, 0)
	err := db.Model(&GroupSupportAwardMgr{}).Where("period >= ?", gsa.Period).Find(&result).Error
	if err != nil {
		return nil, err
	}
	return result, nil
}

//添加记录
func AddGroupSupportAward(model *domain.Model, groupUid string, issuerUserId mysql.ID, resGroupSupportId mysql.ID, userIds []mysql.ID, period string) (*GroupSupportAwardAdmin, []GroupSupportAwardMgr, error) {
	//资源获取
	resGroupSupport, err := res_m.GetResGroupSupportById(model, resGroupSupportId)
	if err != nil {
		return nil, nil, err
	}
	if int(resGroupSupport.MgrNum) < len(userIds) {
		return nil, nil, myerr.NewSysErrorF("AddGroupSupportAward mgrNum:%v 同 len(userIds)=%v 不一致", resGroupSupport.MgrNum, len(userIds))
	}

	//增加群主奖励
	groupSupportAwardAdmin := GroupSupportAwardAdmin{
		Model:             model,
		GroupUid:          groupUid,
		IssuerUserId:      issuerUserId,
		UserId:            issuerUserId,
		DiamondNum:        resGroupSupport.AdminAward,
		Grade:             resGroupSupport.Grade,
		ResGroupSupportId: resGroupSupport.ID,
		Period:            period,
	}

	groupSupportAwardMgrs := make([]GroupSupportAwardMgr, 0, len(userIds))

	//增加管理人奖励
	for i, _ := range userIds {
		groupSupportAwardMgrs = append(groupSupportAwardMgrs, GroupSupportAwardMgr{
			Model:             model,
			GroupUid:          groupUid,
			IssuerUserId:      issuerUserId,
			UserId:            userIds[i],
			DiamondNum:        resGroupSupport.MgrAward,
			Grade:             resGroupSupport.Grade,
			ResGroupSupportId: resGroupSupport.ID,
			Period:            period,
		})
	}
	return &groupSupportAwardAdmin, groupSupportAwardMgrs, nil
}
