package group_r

import (
	"encoding/json"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"hilo-group/_const/enum/diamond_e"
	"hilo-group/_const/enum/group_e"
	"hilo-group/_const/enum/task_e"
	"hilo-group/cv/diamond_cv"
	"hilo-group/cv/group_cv"
	"hilo-group/cv/mic_cv"
	"hilo-group/domain/cache/mic_c"
	"hilo-group/domain/model/diamond_m"
	"hilo-group/domain/model/group_m"
	"hilo-group/domain/model/res_m"
	"hilo-group/domain/model/task_m"
	"hilo-group/domain/service/group_mic_s"
	"hilo-group/myerr"
	"hilo-group/myerr/bizerr"
	"hilo-group/req"
	"hilo-group/resp"
	"strconv"
	"time"
)

// @Tags 群组
// @Summary 麦位所有信息（支持高频获取，前端可以高频获取，进行校正）, v2.5.0_2_211025 版本之前，只能获取5个麦位，兼容于旧版本数据
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid query string true "groupUuid"
// @Success 200 {object} group_cv.CvMic
// @Router /v1/imGroup/mic/all [get]
func GroupMicAllInfoFive(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.Query("groupUuid")

	userId, extendId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	group_m.UpdateMicExpire(model, groupUuid, extendId)
	//加入房间 || 更新房间内有人信息
	//group_m.RoomLivingIn(model, groupUuid, userId)
	//降低频率
	group_m.RoomLivingExpire(model, groupUuid, userId)

	mics, micUsers, err := group_m.GetAllMic(groupUuid, group_e.FiveMicNumType)
	if err != nil {
		return myContext, err
	}
	cvMics, err := group_cv.GetGroupMicAll(mics, micUsers)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, cvMics)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位所有信息（支持高频获取，前端可以高频获取，进行校正）（v2.5.0_2_211025 版本之后，根据类型返回5或者10个麦位信息，数量决定了类型）
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid query string true "groupUuid"
// @Success 200 {object} group_cv.CvMic
// @Router /v1/imGroup/mic/all/type [get]
func GroupMicAllInfoTen(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.Query("groupUuid")

	userId, extendId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)

	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	group_m.UpdateMicExpire(model, groupUuid, extendId)

	//加入房间 || 更新房间内有人信息
	//group_m.RoomLivingIn(model, groupUuid, userId)
	//降低频率
	group_m.RoomLivingExpire(model, groupUuid, userId)

	micNumType, err := group_m.GetMicNumType(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	//兼容于旧版本
	n := group_m.GetMicNum(micNumType)
	if n >= 10 {
		micNumType = group_e.TenMicNumType
	} else {
		micNumType = group_e.FiveMicNumType
	}

	mics := []group_m.Mic{}
	micUsers := []group_m.MicUser{}
	mics, micUsers, err = group_m.GetAllMic(groupUuid, micNumType)
	if err != nil {
		return myContext, err
	}
	cvMics, err := group_cv.GetGroupMicAll(mics, micUsers)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, cvMics)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位所有信息（支持高频获取，前端可以高频获取，进行校正）（v2.5.0_2_211025 版本之后，根据类型返回5或者10个麦位信息，数量决定了类型）
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid query string true "groupUuid"
// @Success 200 {object} group_cv.CvMic
// @Router /v1/imGroup/mic/all/type/new [get]
func GroupMicAllInfoType(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.Query("groupUuid")

	userId, extendId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	group_m.UpdateMicExpire(model, groupUuid, extendId)

	//加入房间 || 更新房间内有人信息
	//group_m.RoomLivingIn(model, groupUuid, userId)
	//降低频率
	group_m.RoomLivingExpire(model, groupUuid, userId)

	micNumType, err := group_m.GetMicNumType(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	var mics []group_m.Mic
	var micUsers []group_m.MicUser

	mics, micUsers, err = group_m.GetAllMic(groupUuid, micNumType)
	if err != nil {
		return myContext, err
	}
	cvMics, err := group_cv.GetGroupMicAll(mics, micUsers)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, cvMics)
	return myContext, nil
}

// @Tags 群组
// @Summary 修改麦位数量
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Param micNumType formData int true "麦位数量类型 1：5个麦位 2：10个麦位, 当：micOn=fales, micNumType值不采纳，传的是0"  //当：micOn=fales, micNumType值不采纳：这是前端要求的
// @Param micOn formData bool true "是否打开麦位"
// @Success 200
// @Router /v1/imGroup/mic/num [put]
func GroupMicNumChange(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, nil
	}
	groupId := c.PostForm("groupId")
	if groupId == "" {
		return myContext, myerr.NewSysError("参数micNumType 不能为空")
	}

	model := domain.CreateModelContext(myContext)
	groupId, err = group_m.ToImGroupId(model, groupId)
	if err != nil {
		return myContext, err
	}

	micNumType, err := strconv.ParseUint(c.PostForm("micNumType"), 10, 8)
	if err != nil {
		return myContext, myerr.NewSysError("参数groupId 不能为空")
	}

	micOn, err := strconv.ParseBool(c.PostForm("micOn"))
	if err != nil {
		return myContext, myerr.NewSysError("参数micOn 有误")
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicNumChange(groupId, userId, group_e.GroupMicNumType(micNumType), micOn); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 查询麦位数量。返回的是枚举。
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId query string true "群ID"
// @Success 200
// @Router /v1/imGroup/mic/num [get]
func GroupMicNum(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	txGroupId := c.Query("groupId")

	groupInfo, err := group_m.GetInfoByTxGroupId(domain.CreateModelContext(myContext), txGroupId)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, groupInfo.MicNumType)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦上表情信令
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Param micEmojiId formData int true "麦上表情ID"
// @Success 200
// @Router /v1/imGroup/mic/emoji/msg [post]
func GroupSendMicSystemMsg(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	_, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, nil
	}

	txGroupId := c.PostForm("groupId")
	if txGroupId == "" {
		return myContext, myerr.NewSysError("参数groupId 不能为空")
	}

	micEmojiId, err := strconv.ParseUint(c.PostForm("micEmojiId"), 10, 64)
	if err != nil {
		return myContext, myerr.WrapErr(err)
	}
	//
	resMicEmoji := res_m.ResMicEmoji{}
	if err := mysql.Db.Model(&res_m.ResMicEmoji{}).First(&resMicEmoji, micEmojiId).Error; err != nil {
		return myContext, myerr.WrapErr(err)
	}
	if resMicEmoji.Status == mysql.NOUSER {
		return myContext, myerr.NewSysError("表情已下架")
	}
	buf, err := json.Marshal(mic_cv.CvMicEmoji{
		Id:      resMicEmoji.ID,
		Name:    resMicEmoji.Name,
		IconUrl: resMicEmoji.IconUrl,
		SvagUrl: resMicEmoji.SvagUrl,
	})
	if err != nil {
		return myContext, myerr.WrapErr(err)
	}
	//
	model := domain.CreateModelContext(myContext)

	if err := group_m.MicRPush(model, txGroupId, group_m.GroupSystemMsg{
		MsgId:   group_e.GroupMicEmoji,
		Source:  externalId,
		Content: string(buf),
	}); err != nil {
		return myContext, err
	}

	resp.ResponseOk(c, nil)
	return myContext, nil
}

type ReturnFlag struct {
	Flag bool `json:"flag"`
	//1:邀请上麦 2：未完成任务上麦任务
	Type uint `json:"type"`
	//钻石数量（2：未完成任务上麦任务才有值）
	DiamondNum uint32 `json:"diamondNum"`
}

// @Tags 群组
// @Summary 邀请上麦校验弹窗
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Success 200 {object} ReturnFlag
// @Router /v1/imGroup/mic/in/invite/dialog [post]
func GroupMicInInviteDialog(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	_, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, nil
	}

	groupId := c.PostForm("groupId")
	if len(groupId) <= 0 {
		return myContext, bizerr.InvalidParameter
	}

	model := domain.CreateModelContext(myContext)
	groupInfo, err := group_m.GetInfoByTxGroupId(model, groupId)
	if err != nil {
		return nil, err
	}
	groupId = groupInfo.ImGroupId

	mics, micUsers, err := group_m.GetAllMic(groupId, groupInfo.MicNumType)
	if err != nil {
		return myContext, err
	}
	cvMics, err := group_cv.GetGroupMicAll(mics, micUsers)
	//用户没有上麦
	micInFlag := false
	//群组麦上有人
	micHasUserFlag := false
	//有空余麦位
	micSpareFlag := false

	for _, r := range cvMics {
		//用户在麦上
		if r.ExternalId != nil && *(r.ExternalId) == externalId {
			micInFlag = true
		}
		//麦上有人
		if r.ExternalId != nil && *(r.ExternalId) != externalId {
			micHasUserFlag = true
		}
		if r.ExternalId == nil && r.Lock == false {
			micSpareFlag = true
		}
	}

	if micInFlag == false && micHasUserFlag == true && micSpareFlag == true {
		resp.ResponseOk(c, ReturnFlag{Flag: true})
		return myContext, nil
	} else {
		resp.ResponseOk(c, ReturnFlag{
			Flag:       false,
			Type:       1,
			DiamondNum: 0,
		})
		return myContext, nil
	}
}

// @Tags 群组
// @Summary 邀请上麦钻石校验弹窗
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Success 200 {object} ReturnFlag
// @Router /v1/imGroup/mic/task/invite/dialog [post]
func GroupMicTaskInviteDialog(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, nil
	}

	groupId := c.PostForm("groupId")
	if groupId == "" {
		return myContext, myerr.NewSysError("参数groupId 不能为空")
	}

	model := domain.CreateModelContext(myContext)

	groupInfo, err := group_m.GetInfoByTxGroupId(model, groupId)
	if err != nil {
		return myContext, err
	}
	if groupInfo == nil {
		return myContext, bizerr.GroupNotFound
	}
	groupId = groupInfo.ImGroupId

	mics, micUsers, err := group_m.GetAllMic(groupId, groupInfo.MicNumType)
	if err != nil {
		return myContext, err
	}
	cvMics, err := group_cv.GetGroupMicAll(mics, micUsers)
	//用户没有上麦
	micInFlag := false
	//群组麦上有人
	micHasUserFlag := false
	//有空余麦位
	micSpareFlag := false

	for _, r := range cvMics {
		//用户在麦上
		if r.ExternalId != nil && *(r.ExternalId) == externalId {
			micInFlag = true
		}
		//麦上有人
		if r.ExternalId != nil && *(r.ExternalId) != externalId {
			micHasUserFlag = true
		}
		if r.ExternalId == nil && r.Lock == false {
			micSpareFlag = true
		}
	}

	if micInFlag == false && micHasUserFlag == true && micSpareFlag == true {
		// 今天展示过了
		if show, _ := mic_c.IsMicDayInviteDialogShowToday(model, userId); show {
			resp.ResponseOk(c, ReturnFlag{
				Flag: false,
			})
			return myContext, nil
		}

		//产品说：同是否能领取了每日钻石关联起来
		beginTime := utils.GetZeroTime(time.Now())
		endTime := beginTime.AddDate(0, 0, 1)

		var n int64
		if err := mysql.Db.Model(&diamond_m.DiamondAccountDetail{}).Where(&diamond_m.DiamondAccountDetail{
			UserId:    userId,
			AddReduce: mysql.ADD,
		}).Where("Created_Time <= ? and Created_Time >= ?", endTime, beginTime).Where("(Operate_Type = ? or Operate_Type = ?)", diamond_e.DailyInAppVip, diamond_e.DailyInAppCommon).Count(&n).Error; err != nil {
			if err != nil {
				return myContext, nil
			}
		}
		if n > 0 {
			//判断是否完成上麦任务
			taskConfig := task_m.TaskConfig{}
			if err := mysql.Db.Model(&task_m.TaskConfig{}).Where(&task_m.TaskConfig{
				Type: task_e.MicIn,
			}).First(&taskConfig).Error; err != nil {
				if err == gorm.ErrRecordNotFound {
				} else {
					return myContext, myerr.WrapErr(err)
				}
			}
			//任务还存在
			if taskConfig.ID != 0 {
				var n int64
				if err := mysql.Db.Model(&task_m.TaskUser{}).Where(&task_m.TaskUser{
					TaskConfigId: taskConfig.ID,
					UserId:       userId,
					DayStr:       time.Now().Format(utils.COMPACT_DATE_FORMAT),
					HasFinish:    mysql.YES,
				}).Count(&n).Error; err != nil {
					return myContext, myerr.WrapErr(err)
				}
				//存在还没完成
				if n == 0 {
					resp.ResponseOk(c, ReturnFlag{
						Flag:       true,
						Type:       2,
						DiamondNum: taskConfig.Diamond,
					})
					return myContext, nil
				}
			}
		}
		resp.ResponseOk(c, ReturnFlag{
			Flag: true,
			Type: 1,
		})
		return myContext, nil
	} else {
		resp.ResponseOk(c, ReturnFlag{
			Flag:       false,
			Type:       1,
			DiamondNum: 0,
		})
		return myContext, nil
	}
}

// @Tags 群组
// @Summary 上麦
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string false "麦序, 空字符串，代表有麦位就上"
// @Success 200
// @Router /v1/imGroup/mic/in [post]
func GroupMicIn(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	nonce := c.GetHeader("nonce")
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupInfo, err := group_m.GetInfoByTxGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}
	if groupInfo == nil {
		return myContext, bizerr.GroupNotFound
	}
	groupUuid = groupInfo.ImGroupId

	// 游客，且设置了游客不能上麦，则失败
	// nonce = hilo,服务端内部调用,机器人
	if groupInfo.TouristMic != 1 && nonce != "hilo" {
		role, err := GetGroupRoleById(model, groupInfo.ImGroupId, userId)
		if err != nil {
			return myContext, err
		}
		if role == group_e.GROUP_VISITOR {
			// 旧版本(2.32.0以下)，提示升级
			_, major, minor, _, err := req.GetAppVersion(c)
			if err != nil {
				return myContext, err
			}
			if major <= 2 && minor < 32 {
				return myContext, bizerr.UpgradeRequired
			}
			return myContext, bizerr.GroupMicBanTourist
		}
	}
	iStr := c.PostForm("i")
	micIndex := -1
	// 用户已经在麦上，必须先下麦！
	micUser, err := group_m.GetMicUserByExternalId(model, externalId)
	if err != nil {
		return myContext, err
	}
	if micUser != nil {
		if iStr == "" {
			// 非切麦操作
			return myContext, bizerr.GroupMicUserHasIn
		}
		// 切换麦位,先下麦
		if err := group_mic_s.NewGroupPowerService(myContext).GroupMicLeave(groupUuid, micUser.I, userId, externalId); err != nil {
			return myContext, err
		}
	}

	if iStr == "" {
		//判断群组设置上的麦 是否被关闭
		if groupInfo.MicOn == false {
			return myContext, bizerr.GroupInfoMicClosed
		}
		//micNum := 5
		//if groupInfo.MicNumType == group_enum.TenMicNumType {
		//	micNum = 10
		//}
		micNum := group_m.GetMicNum(groupInfo.MicNumType)
		for i := 1; i <= micNum; i++ {
			if err := group_mic_s.NewGroupPowerService(myContext).GroupMicIn(groupUuid, i, userId, externalId); err != nil {
				if i == micNum {
					//最后一次了，依旧错误，则
					return myContext, bizerr.GroupMicInByInviteFail
				}
			} else {
				//成功则跳出
				micIndex = i
				break
			}
		}
	} else {
		i, err := strconv.Atoi(iStr)
		if err != nil {
			return myContext, err
		}
		if err := group_mic_s.NewGroupPowerService(myContext).GroupMicIn(groupUuid, i, userId, externalId); err != nil {
			return myContext, err
		}
		micIndex = i
	}
	data := map[string]interface{}{
		"micIndex": micIndex,
	}
	if nonce == "hilo" {
		resp.ResponseOk(c, data)
	} else {
		resp.ResponseOk(c, nil)
	}
	// check cp 麦位置
	go group_mic_s.NewGroupPowerService(myContext).CheckCpOnMic(groupUuid)
	return myContext, nil
}

func GetGroupRoleById(model *domain.Model, imGroupId string, userId uint64) (role group_e.GroupRoleType, err error) {
	role = group_e.GROUP_VISITOR
	roles, _, err := group_m.GetRolesInGroup(model, imGroupId)
	if err != nil {
		return
	}
	for u, r := range roles {
		if u == userId {
			role = r
			return
		}
	}

	isGroupMember, err := group_m.IsGroupMember(model.Db, imGroupId, userId)
	if err != nil {
		return
	}
	if isGroupMember {
		role = group_e.GROUP_MEMBER
	}
	return
}

// @Tags 群组
// @Summary 邀请上麦
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param beInvitedExternalId formData string false "被邀请人的ExternalId"
// @Success 200
// @Router /v1/imGroup/mic/invite [post]
func GroupMicInvite(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	beInvitedExternalId := c.PostForm("beInvitedExternalId")
	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicInvite(groupUuid, userId, beInvitedExternalId); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 自己离开麦位， 管理人让他下麦
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/leave [post]
func GroupMicLeave(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupId := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	_, lang, err := req.GetUserIdLang(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupId, err = group_m.ToImGroupId(model, groupId)
	if err != nil {
		return myContext, err
	}

	micUser, err := group_m.GetMicUser(model, groupId, i)
	if err != nil {
		return myContext, err
	}
	if micUser != nil && micUser.UserId != userId { // 抱下麦
		// 检查权限
		if err = CheckOptToSvip6(model, userId, micUser.UserId, lang, 11); err != nil {
			return myContext, err
		}
	}
	// check cp麦位置
	group_mic_s.NewGroupPowerService(myContext).CheckCpLeaveMic(groupId, userId)

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicLeave(groupId, i, userId, externalId); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位加锁
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/lock [post]
func GroupMicLock(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicLock(userId, externalId, groupUuid, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位解锁
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/unlock [post]
func GroupMicUnLock(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicUnLock(userId, externalId, groupUuid, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 开麦（可以说话）
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/speech/open [post]
func GroupMicSpeechOpen(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}

	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicSpeechOpen(userId, externalId, groupUuid, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 关麦（不可以说话），管理人禁言这个麦位
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/speech/close [post]
func GroupMicSpeechClose(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}

	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	_, lang, err := req.GetUserIdLang(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicSpeechClose(userId, externalId, groupUuid, lang, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位解除静音(打开座位)
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/unmute [post]
func GroupMicUnmute(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicUnMute(userId, externalId, groupUuid, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦位静音(关闭座位)
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupUuid formData string true "groupUuid"
// @Param i formData string true "麦序"
// @Success 200
// @Router /v1/imGroup/mic/mute [post]
func GroupMicMute(c *gin.Context) (*mycontext.MyContext, error) {
	// todo
	myContext := mycontext.CreateMyContext(c.Keys)
	groupUuid := c.PostForm("groupUuid")
	i, err := strconv.Atoi(c.PostForm("i"))
	if err != nil {
		return myContext, err
	}
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}

	model := domain.CreateModelContext(myContext)
	groupUuid, err = group_m.ToImGroupId(model, groupUuid)
	if err != nil {
		return myContext, err
	}

	if err := group_mic_s.NewGroupPowerService(myContext).GroupMicMute(userId, externalId, groupUuid, i); err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, nil)
	return myContext, nil
}

// @Tags 群组
// @Summary 麦上用户群发成员广播
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Param content formData string true "内容"
// @Success 200 {object} diamond_cv.CvDiamond
// @Router /v1/imGroup/mic/mass [post]
func GroupMicMass(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}
	groupId := c.PostForm("groupId")
	if len(groupId) <= 0 {
		return myContext, bizerr.InvalidParameter
	}

	model := domain.CreateModelContext(myContext)
	groupId, err = group_m.ToImGroupId(model, groupId)
	if err != nil {
		return myContext, err
	}

	content := c.PostForm("content")

	if err := group_mic_s.NewGroupPowerService(myContext).GroupIMMassByInMic(groupId, userId, externalId, content); err != nil {
		return myContext, err
	}
	diamond, err := diamond_cv.GetDiamond(userId)
	if err != nil {
		return nil, myerr.WrapErr(err)
	}
	resp.ResponseOk(c, diamond)
	return myContext, nil
}

// @Tags 群组
// @Summary 用户群发成员广播
// @Accept application/x-www-form-urlencoded
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupId formData string true "群ID"
// @Param content formData string true "内容"
// @Success 200 {object} diamond_cv.CvDiamond
// @Router /v1/imGroup/mgr/mass [post]
func GroupMgrMass(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, externalId, err := req.GetUserIdAndExtId(c, myContext)
	if err != nil {
		return myContext, err
	}
	groupId := c.PostForm("groupId")
	if len(groupId) <= 0 {
		return myContext, bizerr.InvalidParameter
	}

	model := domain.CreateModelContext(myContext)
	groupId, err = group_m.ToImGroupId(model, groupId)
	if err != nil {
		return myContext, err
	}

	content := c.PostForm("content")

	if err := group_mic_s.NewGroupPowerService(myContext).GroupIMMassByMgr(groupId, userId, externalId, content); err != nil {
		return myContext, err
	}
	diamond, err := diamond_cv.GetDiamond(userId)
	if err != nil {
		return nil, myerr.WrapErr(err)
	}
	resp.ResponseOk(c, diamond)
	return myContext, nil
}
