package group_c

import (
	"context"
	"git.hilo.cn/hilo-common/resource/redisCli"
	"hilo-group/_const/redis_key"
	"strings"
	"time"
)

func GetGroupMemberCount(groupId string) (int, error) {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Get(context.Background(), key).Int()
}

func SetGroupMemberCount(groupId string, count uint, ttl time.Duration) error {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Set(context.Background(), key, count, ttl).Err()
}

func ClearGroupMemberCount(groupId string) error {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Del(context.Background(), key).Err()
}

func getGroupMemberKey(groupId string) string {
	return strings.Replace(redis_key.GroupMemberPrefix, "{groupId}", groupId, -1)
}

func getGroupConsumeKey(groupId string) string {
	return strings.Replace(redis_key.GroupConsumePrefix, "{groupId}", groupId, -1)
}

func GetGroupMember(groupId string) ([]string, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SMembers(context.Background(), key).Result()
}

func GetGroupMemberCard(groupId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SCard(context.Background(), key).Result()
}

func GetGroupMemberAsMap(groupId string) (map[string]struct{}, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SMembersMap(context.Background(), key).Result()
}

func SetExists(groupId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.Exists(context.Background(), key).Result()
}

func AddGroupMember(groupId string, extIds []string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SAdd(context.Background(), key, extIds).Result()
}

func RemoveGroupMember(groupId string, externalId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SRem(context.Background(), key, externalId).Result()
}

func SetGroupMemberTTL(groupId string, ttl time.Duration) (bool, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.Expire(context.Background(), key, ttl).Result()
}

func GetGroupConsume(groupId string) (uint64, error) {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Get(context.Background(), key).Uint64()
}

func SetGroupConsume(groupId string, consume uint64, ttl time.Duration) error {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Set(context.Background(), key, consume, ttl).Err()
}

func ClearGroupConsume(groupId string) error {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Del(context.Background(), key).Err()
}
