package group_s

import (
	"git.hilo.cn/hilo-common/domain"
	"hilo-group/_const/enum/gift_e"
	"hilo-group/domain/event/group_ev"
	"hilo-group/domain/model/gift_m"
	"hilo-group/domain/model/group_m"
	"hilo-group/domain/model/res_m"
	"hilo-group/domain/model/user_m"
	"hilo-group/myerr/bizerr"
	"time"
)

// 群组支持名单过滤
func (s *GroupService) GroupSupportList(groupId string, uids []uint64) ([]uint64, []uint64, error) {
	if len(uids) <= 0 {
		return uids, nil, nil
	}

	result := make([]uint64, 0)
	out := make([]uint64, 0)

	err := s.svc.Transactional(func() error {
		model := domain.CreateModel(s.svc.CtxAndDb)

		// 1. 去掉非群管理者
		roles, _, err := group_m.GetRolesInGroup(model, groupId)
		if err != nil {
			return err
		}
		userIds := make([]uint64, 0)
		for _, i := range uids {
			if _, ok := roles[i]; ok {
				userIds = append(userIds, i)
			} else {
				out = append(out, i)
				model.Log.Infof("GroupSupportList: rule out %d, no role", i)
			}
		}

		// TODO: 去掉非群成员

		//（4）1个账户只能做1个群组的管理员（5）1个设备下只允许领取1个管理奖励
		_, _, period := group_m.GetLastSupportPeriod(time.Now())
		gsa := group_m.GroupSupportAwardMgr{Period: period}
		rows, err := gsa.Get(model.Db)
		if err != nil {
			return err
		}
		awards := make(map[uint64]struct{}, 0)
		for _, i := range rows {
			awards[i.UserId] = struct{}{}
		}

		uids = userIds
		userIds = make([]uint64, 0)
		m := make(map[uint64]uint64)
		for _, u := range uids {
			m, err := user_m.GetSameImeiMap(model, u)
			if err != nil {
				return err
			}

			passed := true
			for _, i := range m {
				if _, ok := awards[i]; ok {
					if i == u {
						passed = false
						model.Log.Infof("GroupSupportList: rule out %d, already awarded", i)
					} else {
						passed = false
						model.Log.Infof("GroupSupportList: rule out %d, imei awarded", i)
					}
				}
			}
			if passed == true {
				userIds = append(userIds, u)
			} else {
				out = append(out, u)
			}
		}
		model.Log.Infof("GroupSupportList: uids %v, map %v", uids, m)

		_, supportLevel, err := s.GetSupportLevel(groupId)
		if err != nil {
			return err
		}

		if uint32(len(userIds)) > supportLevel {
			model.Log.Infof("GroupSupportList: rule out %v, limit exeeded", userIds[supportLevel:])
			out = append(out, userIds[supportLevel:]...)
			userIds = userIds[0:supportLevel]
		}
		result = userIds
		return nil
	})

	if err == nil {
		return result, out, nil
	} else {
		return nil, nil, err
	}
}

func (s *GroupService) GetSupportLevel(groupId string) (uint64, uint32, error) {
	model := domain.CreateModel(s.svc.CtxAndDb)

	beginTime, endTime, _ := group_m.GetLastSupportPeriod(time.Now())

	g := gift_m.GiftOperate{SceneType: gift_e.GroupSceneType, SceneUid: groupId, Model: model}
	count, consume, err := g.GetConsumeByRange(beginTime, endTime)
	if err != nil {
		return 0, 0, err
	}

	rec, err := res_m.GetResGroupSupportBy(model, count, consume)
	if err != nil {
		return 0, 0, err
	}
	if rec != nil {
		return rec.ID, rec.MgrNum, nil
	}
	return 0, 0, nil
}

//群组支持奖励
func (s *GroupService) GroupSupportAward(groupId string, profitAllocator uint64, userIds []uint64, resId uint64, period string) error {
	return s.svc.Transactional(func() error {
		model := domain.CreateModel(s.svc.CtxAndDb)
		//
		groupInfo, err := group_m.GetGroupInfo(model, groupId)
		if groupInfo == nil {
			return bizerr.GroupNotFound
		}
		//发放奖励
		groupSupportAwardAdmin, groupSupportAwardMgrs, err := group_m.AddGroupSupportAward(model, groupId, profitAllocator, resId, userIds, period)
		if err != nil {
			return err
		}

		if err := groupSupportAwardAdmin.Persistent(); err != nil {
			return err
		}

		groupSupportEvent := group_ev.InitGroupSupportEvent(len(groupSupportAwardMgrs), groupInfo.Code)
		//数据持久化
		groupSupportEvent.AddAdmin(groupSupportAwardAdmin.ID, groupSupportAwardAdmin.UserId, groupSupportAwardAdmin.DiamondNum)
		for i, _ := range groupSupportAwardMgrs {
			if err := groupSupportAwardMgrs[i].Persistent(); err != nil {
				return err
			}
			groupSupportEvent.AddMgr(groupSupportAwardMgrs[i].ID, groupSupportAwardMgrs[i].UserId, groupSupportAwardMgrs[i].DiamondNum)
		}
		return group_ev.PublishGroupSupport(model, groupSupportEvent)
	})
}

func (s *GroupService) RenewGroupSupporter(groupId string, userIds []uint64) error {
	return s.svc.Transactional(func() error {
		model := domain.CreateModel(s.svc.CtxAndDb)
		gs := group_m.GroupSupporter{GroupId: groupId}
		if err := gs.Delete(model.Db); err != nil {
			return err
		}

		if len(userIds) > 0 {
			gs = group_m.GroupSupporter{GroupId: groupId}
			if err := gs.BatchSave(model.Db, userIds); err != nil {
				return err
			}
		}
		return nil
	})
}
