package group_power_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"github.com/gin-gonic/gin"
	"github.com/jinzhu/now"
	"hilo-group/_const/enum/groupPower_e"
	"hilo-group/cv/group_power_cv"
	"hilo-group/cv/user_cv"
	"hilo-group/domain/model/groupPower_m"
	"hilo-group/domain/model/user_m"
	"hilo-group/myerr/bizerr"
	"hilo-group/req"
	"hilo-group/resp"
	"strconv"
	"time"
)

// @Tags 国家势力
// @Summary 家族榜单
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param period path string true "榜单周期 day:日 week:周 month:月"
// @Success 200 {object} []group_power_cv.CvGroupPowerRank
// @Router /v1/groupPower/rank/{period} [get]
func GroupPowerRank(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}
	period := c.Param("period")
	if period != "day" && period != "week" && period != "month" {
		return myContext, bizerr.InvalidParameter
	}
	var beginDate, endDate string
	switch period {
	case "day":
		beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
	case "week":
		beginDate, endDate = utils.GetMonday(time.Now()).Format("2006-01-02"), utils.GetMonday(time.Now()).AddDate(0, 0, 6).Format("2006-01-02")
	case "month":
		beginDate, endDate = now.BeginningOfMonth().Format("2006-01-02"), now.EndOfMonth().Format("2006-01-02")
	}
	var model = domain.CreateModelContext(myContext)
	rank, err := groupPower_m.GetGroupPowerExpRank(model, beginDate, endDate, 30)
	if err != nil {
		return myContext, err
	}
	var ids []mysql.ID
	for _, g := range rank {
		ids = append(ids, g.GroupPowerId)
	}
	response := group_power_cv.CvGroupPowerRank{}
	myGroupPower, err := groupPower_m.GetGroupPowerUserOrNil(model, userId)
	if err != nil {
		return myContext, err
	}
	if myGroupPower != nil {
		ids = append(ids, myGroupPower.GroupPowerId)
	}
	grades, err := groupPower_m.MGetGroupPowerGrade(model, ids)
	if err != nil {
		return myContext, err
	}
	groupPowers, err := groupPower_m.MGetGroupPowerInfoMap(model, ids)
	if err != nil {
		return myContext, err
	}
	if myGroupPower != nil {
		myGroupPowerRank, err := groupPower_m.GetMyGroupPowerExpRank(model, beginDate, endDate, myGroupPower.GroupPowerId)
		if err != nil {
			return myContext, err
		}
		response.MyGroupPower = &group_power_cv.CvGroupPowerRankData{
			CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
				Id:        myGroupPower.GroupPowerId,
				Icon:      groupPowers[myGroupPower.GroupPowerId].Icon,
				Name:      groupPowers[myGroupPower.GroupPowerId].Name,
				Nameplate: groupPowers[myGroupPower.GroupPowerId].Nameplate,
			},
			CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
				Grade: grades[myGroupPower.GroupPowerId].Grade,
				Exp:   myGroupPowerRank.Exp,
			},
			Rank: 0,
		}
	}
	for _, v := range rank {
		if response.MyGroupPower != nil && v.GroupPowerId == response.MyGroupPower.Id {
			response.MyGroupPower.Rank = v.Rank
		}
		response.Items = append(response.Items, group_power_cv.CvGroupPowerRankData{
			CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
				Id:        v.GroupPowerId,
				Icon:      groupPowers[v.GroupPowerId].Icon,
				Name:      groupPowers[v.GroupPowerId].Name,
				Nameplate: groupPowers[v.GroupPowerId].Nameplate,
			},
			CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
				Grade: grades[v.GroupPowerId].Grade,
				Exp:   v.Exp,
			},
			Rank: v.Rank,
		})
	}
	if response.MyGroupPower != nil && response.MyGroupPower.Rank == 0 {
		response.MyGroupPower.Rank = 31 // 客户端统一显示30+
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族日周月榜单top3
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Success 200 {object} map[string][]group_power_cv.CvGroupPowerRankData
// @Router /v1/groupPower/rankTop [get]
func GroupPowerRankTop(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	periods := []string{"day", "week", "month"}
	var response = make(map[string][]group_power_cv.CvGroupPowerRankData)
	for _, period := range periods {
		var beginDate, endDate string
		switch period {
		case "day":
			beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
		case "week":
			beginDate, endDate = utils.GetMonday(time.Now()).Format("2006-01-02"), utils.GetMonday(time.Now()).AddDate(0, 0, 6).Format("2006-01-02")
		case "month":
			beginDate, endDate = now.BeginningOfMonth().Format("2006-01-02"), now.EndOfMonth().Format("2006-01-02")
		}
		var model = domain.CreateModelContext(myContext)
		rank, err := groupPower_m.GetGroupPowerExpRank(model, beginDate, endDate, 3)
		if err != nil {
			return myContext, err
		}
		var ids []mysql.ID
		for _, g := range rank {
			ids = append(ids, g.GroupPowerId)
		}

		grades, err := groupPower_m.MGetGroupPowerGrade(model, ids)
		if err != nil {
			return myContext, err
		}
		groupPowers, err := groupPower_m.MGetGroupPowerInfoMap(model, ids)
		if err != nil {
			return myContext, err
		}
		for _, v := range rank {
			response[period] = append(response[period], group_power_cv.CvGroupPowerRankData{
				CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
					Id:        v.GroupPowerId,
					Icon:      groupPowers[v.GroupPowerId].Icon,
					Name:      groupPowers[v.GroupPowerId].Name,
					Nameplate: groupPowers[v.GroupPowerId].Nameplate,
				},
				CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
					Grade: grades[v.GroupPowerId].Grade,
					Exp:   v.Exp,
				},
				Rank: v.Rank,
			})
		}
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

type GroupPowerStarReq struct {
	GroupPowerId mysql.ID                        `form:"groupPowerId" binding:"required"`
	Type         groupPower_e.GroupPowerStarType `form:"type" binding:"required"`
	PageSize     int                             `form:"pageSize"`
	PageIndex    int                             `form:"pageIndex"`
}

// @Tags 国家势力
// @Summary 家族之星
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupPowerId query int true "家族id"
// @Param type query integer true "类型 1:送礼 2:活跃 3:收礼物"
// @Param pageSize query int false "分页大小 默认：30" default(30)
// @Param pageIndex query int false "第几个分页，从1开始 默认：1" default(1)
// @Success 200 {object} []group_power_cv.CvGroupPowerStarData
// @Router /v1/groupPower/star [get]
func GroupPowerStar(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	param := new(GroupPowerStarReq)
	if err := c.ShouldBindQuery(param); err != nil {
		return myContext, err
	}
	if param.PageIndex <= 0 {
		param.PageIndex = 1
	}
	if param.PageSize <= 0 {
		param.PageSize = 30
	}
	var model = domain.CreateModelContext(myContext)
	offset, limit := (param.PageIndex-1)*param.PageSize, param.PageSize
	rank, err := groupPower_m.GetGroupPowerMonthStarRank(model, param.GroupPowerId, param.Type, offset, limit)
	if err != nil {
		return myContext, err
	}
	var response []group_power_cv.CvGroupPowerStarData
	var userIds []mysql.ID
	for _, row := range rank {
		userIds = append(userIds, row.UserId)
	}
	users, err := user_m.GetUserMapByIds(model, userIds)
	for _, row := range rank {
		user := users[row.UserId]
		score := row.Score
		if param.Type == groupPower_e.GroupPowerStarTypeActive {
			score = score / 60
		}
		if score <= 0 {
			continue
		}
		response = append(response, group_power_cv.CvGroupPowerStarData{
			User: user_cv.CvUserTiny{
				Id:         user.ID,
				ExternalId: user.ExternalId,
				Code:       user.Code,
				Nick:       user.Nick,
				Avatar:     user.Avatar,
			},
			Score: score,
		})
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族之星
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param period path string true "周期 day|week|month"
// @Param groupPowerId query int true "家族id"
// @Param type query integer true "类型 1:送礼 2:活跃 3:收礼物"
// @Param pageSize query int false "分页大小 默认：30" default(30)
// @Param pageIndex query int false "第几个分页，从1开始 默认：1" default(1)
// @Success 200 {object} []group_power_cv.CvGroupPowerStarData
// @Router /v1/groupPower/star/{period} [get]
func GroupPowerStarPeriod(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	param := new(GroupPowerStarReq)
	if err := c.ShouldBindQuery(param); err != nil {
		return myContext, err
	}
	if param.PageIndex <= 0 {
		param.PageIndex = 1
	}
	if param.PageSize <= 0 {
		param.PageSize = 30
	}
	period := c.Param("period")
	if period != "day" && period != "week" && period != "month" {
		return myContext, bizerr.InvalidParameter
	}
	var model = domain.CreateModelContext(myContext)
	offset, limit := (param.PageIndex-1)*param.PageSize, param.PageSize
	rank, err := groupPower_m.GetGroupPowerStarRankPeriod(model, period, param.GroupPowerId, param.Type, offset, limit)
	if err != nil {
		return myContext, err
	}
	var response []group_power_cv.CvGroupPowerStarData
	var userIds []mysql.ID
	for _, row := range rank {
		userIds = append(userIds, row.UserId)
	}
	users, err := user_m.GetUserMapByIds(model, userIds)
	for _, row := range rank {
		user := users[row.UserId]
		score := row.Score
		if param.Type == groupPower_e.GroupPowerStarTypeActive {
			score = score / 60
		}
		if score <= 0 {
			continue
		}
		response = append(response, group_power_cv.CvGroupPowerStarData{
			User: user_cv.CvUserTiny{
				Id:         user.ID,
				ExternalId: user.ExternalId,
				Code:       user.Code,
				Nick:       user.Nick,
				Avatar:     user.Avatar,
			},
			Score: score,
		})
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族等级页
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupPowerId query int true "家族id"
// @Success 200 {object} group_power_cv.CvGroupPowerGradeDetail
// @Router /v1/h5/groupPower/grade/detail [get]
func GroupPowerGradeDetail(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}
	groupPowerId, err := strconv.ParseUint(c.Query("groupPowerId"), 10, 16)
	if err != nil || groupPowerId <= 0 {
		return myContext, bizerr.InvalidParameter
	}
	var model = domain.CreateModelContext(myContext)
	groupPowerInfo, err := groupPower_m.MGetGroupPowerInfoMap(model, []mysql.ID{groupPowerId})
	if err != nil {
		return myContext, err
	}
	gradeM, err := groupPower_m.MGetGroupPowerGrade(model, []mysql.ID{groupPowerId})
	if err != nil {
		return myContext, err
	}
	groupPower, ok := groupPowerInfo[groupPowerId]
	if !ok {
		return myContext, bizerr.GroupNotFound
	}
	grade := gradeM[groupPowerId]
	nextExp := groupPower_e.GroupPowerGradeExp[groupPower_e.GroupPowerGradeMax]
	if grade.Grade != groupPower_e.GroupPowerGradeMax {
		nextExp = groupPower_e.GroupPowerGradeExp[grade.Grade+1]
	}
	expireAt := ""
	if grade.ExpireAt.After(time.Now()) {
		expireAt = grade.ExpireAt.Add(-time.Second).Format("2006-01-02") // 减少一秒格式化天
	}
	if grade.Grade == groupPower_e.GroupPowerGrade0 {
		expireAt = utils.AddDate(now.EndOfMonth(), 0, 1).Add(-time.Second).Format("2006-01-02") // 0级别(黑铁) 没有有效期,给下个月月底
	}
	isMember := false
	if exits, id, err := groupPower_m.CheckGroupPowerUser(model, userId); err != nil {
		return myContext, err
	} else if exits && id == groupPowerId {
		isMember = true
	}
	response := group_power_cv.CvGroupPowerGradeDetail{
		GroupPowerBase: group_power_cv.CvGroupPowerBase{
			Id:        groupPower.ID,
			Icon:      groupPower.Icon,
			Name:      groupPower.Name,
			Nameplate: groupPower.Nameplate,
		},
		GroupPowerGrade: group_power_cv.CvGroupPowerGrade{
			Grade:    grade.Grade,
			Exp:      grade.Exp,
			NextExp:  nextExp,
			ExpireAt: expireAt,
		},
		ResGradeList:  group_power_cv.CvResGradeList,
		PrivilegeList: group_power_cv.GroupPowerGradePrivilegeNum[grade.Grade],
		IsMember:      isMember,
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}
