package room_c

import (
	"context"
	"errors"
	"fmt"
	"git.hilo.cn/hilo-common/mylogrus"
	"git.hilo.cn/hilo-common/resource/redisCli"
	redis2 "github.com/go-redis/redis/v8"
	"hilo-group/_const/redis_key"
	"strconv"
	"time"
)

func ProcessUserRoomVisit(userId uint64, groupId string) error {
	key := redis_key.GetUserEnterRoomKey(userId)
	now := time.Now()
	ret, err := redisCli.GetRedis().ZAdd(context.Background(), key, &redis2.Z{
		Score:  float64(now.Unix()),
		Member: groupId,
	}).Result()
	if err != nil {
		return err
	}
	mylogrus.MyLog.Infof("ProcessUserRoomVisit, ZADD %s, return %d", key, ret)

	return nil
}

// 查询用户访问过的房间及其时间
func GetUserRoomVisit(userId uint64) (map[string]int64, error) {
	key := redis_key.GetUserEnterRoomKey(userId)
	ret, err := redisCli.GetRedis().ZRangeWithScores(context.Background(), key, 0, -1).Result()
	if err != nil {
		return nil, err
	}
	mylogrus.MyLog.Infof("GetUserRoomVisit, ZRangeWithScores %s, return %v", key, ret)

	result := make(map[string]int64, 0)
	for _, i := range ret {
		result[i.Member.(string)] = int64(i.Score)
	}
	return result, nil
}

// 批量获取房间访问人数
func MGetRoomVisitCount(groupIds []string) (map[string]string, error) {
	key := redis_key.GetPrefixRoomVisitCount()
	visit := make(map[string]string)
	if len(groupIds) <= 0 {
		return visit, nil
	}
	res, err := redisCli.GetRedis().HMGet(context.Background(), key, groupIds...).Result()
	if err != nil {
		return visit, err
	}
	if len(res) != len(groupIds) {
		return visit, errors.New(fmt.Sprintf("MGetRoomVisitCount fail,miss match len,%v-%v", len(res), len(groupIds)))
	}
	for i, groupId := range groupIds {
		if cnt, ok := res[i].(string); ok {
			visit[groupId] = cnt
		}
	}
	return visit, nil
}

func GetSetRoomVisitCount(groupId string) (int64, error) {
	key := redis_key.GetPrefixGroupInUserDuration(groupId)
	vc, err := redisCli.GetRedis().ZCard(context.Background(), key).Result()
	// 查到合法值后，更新二级缓存
	if err == nil && vc >= 0 {
		ret, err := saveRoomVisitCount(groupId, vc)
		mylogrus.MyLog.Infof("saveRoomVisitCount %s, ret = %d, err: %v", groupId, ret, err)
	}
	return vc, err
}

func saveRoomVisitCount(groupId string, count int64) (int64, error) {
	key := redis_key.GetPrefixRoomVisitCount()
	return redisCli.GetRedis().HSet(context.Background(), key, groupId, strconv.FormatInt(count, 10)).Result()
}
