package groupPower_c

import (
	"fmt"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"github.com/go-redis/redis/v8"
	"github.com/jinzhu/now"
	"github.com/pkg/errors"
	"github.com/spf13/cast"
	"hilo-group/_const/enum/groupPower_e"
	"hilo-group/_const/redis_key/groupPower_k"
	"time"
)

// 家族之星-排名
type GroupPowerStarRank struct {
	Period       string
	GroupPowerId mysql.ID
	UserId       mysql.ID
	Type         groupPower_e.GroupPowerStarType
	Score        mysql.Num
}

// 增加家族之星分数
func IncrGroupPowerDayStarScore(model *domain.Model, groupPowerId, userId mysql.ID, _type groupPower_e.GroupPowerStarType, score mysql.Num) error {
	ttl := map[string]time.Duration{
		"day":   time.Hour * 24 * 7,
		"week":  time.Hour * 24 * 7 * 30,
		"month": time.Hour * 24 * 7 * 30 * 2,
	}
	for _, period := range []string{"day", "week", "month"} {
		date := ""
		switch period {
		case "day":
			date = time.Now().Format("2006-01-02")
		case "week":
			date = now.BeginningOfWeek().Format("2006-01-02")
		case "month":
			date = now.BeginningOfMonth().Format("2006-01-02")
		}
		key := groupPower_k.GetGroupPowerStarRankKey(_type, period, groupPowerId, date)
		model.RedisCluster.ZIncrBy(model, key, float64(score), fmt.Sprintf("%d", userId))
		model.RedisCluster.Expire(model, key, ttl[period])
	}
	return nil
}

// 获取家族之星排行
// param period: day|week|month
func GetGroupPowerStarRankPeriod(model *domain.Model, period string, groupPowerId mysql.ID, _type groupPower_e.GroupPowerStarType, offset, limit int, targetDate ...string) ([]*GroupPowerStarRank, error) {
	var res []*GroupPowerStarRank
	date := ""
	switch period {
	case "day":
		date = time.Now().Format("2006-01-02")
	case "week":
		date = now.BeginningOfWeek().Format("2006-01-02")
	case "month":
		date = now.BeginningOfMonth().Format("2006-01-02")
	}
	if len(date) <= 0 {
		return res, errors.New("illegal date")
	}
	if len(targetDate) > 0 {
		date = targetDate[0]
	}
	key := groupPower_k.GetGroupPowerStarRankKey(_type, period, groupPowerId, date)
	rows, err := model.RedisCluster.ZRevRangeByScoreWithScores(model, key, &redis.ZRangeBy{
		Min:    "-inf",
		Max:    "+inf",
		Offset: int64(offset),
		Count:  int64(limit),
	}).Result()
	if err != nil {
		model.Log.Errorf("GetGroupPowerStarRankPeriod fail:%v", err)
		return res, err
	}
	for _, v := range rows {
		res = append(res, &GroupPowerStarRank{
			Period:       period,
			GroupPowerId: groupPowerId,
			UserId:       cast.ToUint64(v.Member),
			Type:         _type,
			Score:        mysql.Num(v.Score),
		})
	}
	return res, nil
}

// 获取家族之星三个排行榜的各自第一名
func GetGroupPowerMonthStartTop1(model *domain.Model, groupPowerId mysql.ID) ([]*GroupPowerStarRank, error) {
	var res []*GroupPowerStarRank
	r1, err := GetGroupPowerStarRankPeriod(model, "month", groupPowerId, groupPower_e.GroupPowerStarTypeFamous, 0, 1)
	if err != nil {
		return res, err
	}
	r2, err := GetGroupPowerStarRankPeriod(model, "month", groupPowerId, groupPower_e.GroupPowerStarTypeActive, 0, 1)
	if err != nil {
		return res, err
	}
	r3, err := GetGroupPowerStarRankPeriod(model, "month", groupPowerId, groupPower_e.GroupPowerStarTypeCharm, 0, 1)
	if err != nil {
		return res, err
	}
	res = append(res, r1...)
	res = append(res, r2...)
	res = append(res, r3...)
	return res, nil
}
