package group_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/rpc"
	"hilo-group/domain/model/noble_m"
	"hilo-group/domain/model/user_m"
	"sort"
	"time"
)

type RoomOnlineUser struct {
	Total int                 `json:"total"`
	Users []RoomOnlineUserMsg `json:"users"`
	//客户端对比，只要
	ServiceTime int64 `json:"serviceTime"`
}

type RoomOnlineUserMsg struct {
	Id           uint64 `json:"id,omitempty"`
	ExternalId   string `json:"externalId"`
	Avatar       string `json:"avatar"`
	Nick         string `json:"nick"`
	Sex          uint8  `json:"sex"`
	Code         string `json:"code"`
	Country      string `json:"country"`
	CountryIcon  string `json:"countryIcon"`
	IsPrettyCode bool   `json:"isPrettyCode"` // 是否靓号
	IsLogout     bool   `json:"isLogout"`     //是否注销 true:已经注销， false:没有注销
	//是否VIP用户
	IsVip bool `json:"isVip"`
	//贵族
	Noble RoomUserNoble `json:"noble"` // 当前的
	//VIP用户过期时间(只有自己查询自己，才返回)
	VipExpireTime *int64     `json:"vipExpireTime"`
	SvipLevel     int        `json:"svipLevel"`
	Svip          rpc.CvSvip `json:"svip"`
}

type RoomUserNoble struct {
	Level   uint16 `json:"level"`
	EndTime int64  `json:"endTime"`
}

func GetRoomOnlineUser(cxt *mycontext.MyContext, groupId string) (RoomOnlineUser, error) {
	userIds, total, err := getGroupInUserIds(cxt, groupId)
	if err != nil {
		return RoomOnlineUser{}, err
	}
	var model = domain.CreateModelContext(cxt)
	userMap, err := user_m.GetUserMapByIds(model, userIds)
	if err != nil {
		return RoomOnlineUser{}, err
	}

	vipMap, err := user_m.BatchGetVips(userIds)
	if err != nil {
		return RoomOnlineUser{}, err
	}

	nobleMap, err := noble_m.BatchGetNobleLevel(mysql.Db, userIds)
	if err != nil {
		return RoomOnlineUser{}, err
	}
	svip, _ := rpc.MGetUserSvip(model, userIds)

	users := make([]RoomOnlineUserMsg, 0, len(userIds))
	for i := range userIds {
		user, _ := userMap[userIds[i]]
		users = append(users, RoomOnlineUserMsg{
			Id:           user.ID,
			ExternalId:   user.ExternalId,
			Avatar:       user.Avatar,
			Nick:         user.Nick,
			Sex:          user.Sex,
			Code:         user.Code,
			Country:      user.Country,
			CountryIcon:  user.CountryIcon,
			IsPrettyCode: user.Code != user.OriginCode,
			IsLogout:     false,
			IsVip:        vipMap[userIds[i]] != nil,
			Noble: RoomUserNoble{
				Level:   nobleMap[userIds[i]],
				EndTime: 0,
			},
			VipExpireTime: vipMap[userIds[i]],
			SvipLevel:     svip[userIds[i]].SvipLevel,
			Svip:          rpc.CopySimpleSvip(svip[userIds[i]]),
		})
	}
	return RoomOnlineUser{
		Total:       total,
		Users:       users,
		ServiceTime: time.Now().UnixNano(),
	}, nil
}

var groupOnlineUserIdMaxNum int = 15

//显示的最大数量是15个。
func getGroupInUserIds(cxt *mycontext.MyContext, groupId string) ([]uint64, int, error) {
	userIds, err := RoomLivingExistsUserId(groupId)
	if err != nil {
		return []uint64{}, 0, err
	}
	if len(userIds) == 0 {
		return []uint64{}, 0, nil
	}
	var model = domain.CreateModelContext(cxt)
	svips, err := rpc.MGetUserSvip(model, userIds)
	if err != nil {
		model.Log.Errorf("MGetUserSvip fail :%v", err)
	}
	//贵族
	nobleLevelMap, err := noble_m.BatchGetNobleLevel(mysql.Db, userIds)
	if err != nil {
		model.Log.Errorf("BatchGetNobleLevel fail :%v", err)
	}
	//获取vip
	vipMap, err := user_m.BatchGetVips(userIds)
	if err != nil {
		model.Log.Errorf("MGetUserSvip fail :%v", err)
	}
	// svip > noble > vip
	sort.Slice(userIds, func(i, j int) bool {
		if svips[userIds[i]].SvipLevel > svips[userIds[j]].SvipLevel {
			return true
		} else if svips[userIds[i]].SvipLevel == svips[userIds[j]].SvipLevel {
			if nobleLevelMap[userIds[i]] > nobleLevelMap[userIds[j]] {
				return true
			} else if nobleLevelMap[userIds[i]] == nobleLevelMap[userIds[j]] {
				var vi, vj int64
				if ts, ok := vipMap[userIds[i]]; ok && ts != nil {
					vi = *ts
				}
				if ts, ok := vipMap[userIds[j]]; ok && ts != nil {
					vj = *ts
				}
				return vi > vj
			}
		}
		return false
	})
	total := len(userIds)
	if total > groupOnlineUserIdMaxNum {
		return userIds[0:groupOnlineUserIdMaxNum], total, nil
	}
	return userIds, total, nil
}
