package group_m

import (
	"encoding/json"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/config"
	"git.hilo.cn/hilo-common/rpc"
	"git.hilo.cn/hilo-common/sdk/tencentyun"
	"hilo-group/_const/enum/group_e"
	"hilo-group/domain/model/noble_m"
	"hilo-group/domain/model/user_m"
	"runtime/debug"
	"time"
)

//公屏消息
type PublicScreenMsg struct {
	Code uint8
	Data map[string]string
}

// 信令消息
type GroupSystemMsg struct {
	MsgId   group_e.TypeSignalMsg `json:"msgId"`
	Source  string                `json:"source"`
	Target  string                `json:"target"`
	Content string                `json:"content"`
}

type UserJoinPublicMsg struct {
	Type       uint8  `json:"type"`
	ExternalId string `json:"externalId"`
	Nick       string `json:"nick"`
}

type CommonPublicMsg struct {
	Type               uint8      `json:"type"`
	OperatorExternalId string     `json:"operatorExternalId"`
	OperatorNick       string     `json:"operatorNick"`
	OperatorAvatar     string     `json:"operatorAvatar"`
	ExternalId         string     `json:"externalId"`
	Nick               string     `json:"nick"`
	Avatar             string     `json:"avatar"`
	IsVip              bool       `json:"is_vip"`
	Role               uint16     `json:"role"`
	NobleLevel         uint16     `json:"nobleLevel"`
	SvipLevel          int        `json:"svipLevel"`
	Svip               rpc.CvSvip `json:"svip"`
}

type RollDiceMsg struct {
	CommonPublicMsg
	Dices []int `json:"dices"`
}

type GroupSupportH5 struct {
	CommonPublicMsg
	H5 string `json:"h5"`
}

type RocketAwardMsg struct {
	CommonPublicMsg
	AwardType uint16 `json:"awardType"` // 奖励类型
	Number    uint32 `json:"number"`    // 数量（天/个数）
}

type LuckyboxAwardMsg struct {
	CommonPublicMsg
	Number uint32 `json:"number"` //前端要求，含义钻石数量
}

type FruitMachineAwardMsg struct {
	CommonPublicMsg
	DiamondNum uint `json:"diamondNum"`
}

type GroupCustomMsgContentType = int

const (
	// 跳转url
	GroupCustomMsgH5 GroupCustomMsgContentType = 1
	// 跳转房间
	GroupCustomMsgJump GroupCustomMsgContentType = 2
	//

)

type GroupCustomMsg struct {
	CommonPublicMsg
	ContentType int    `json:"contentType"`
	Content     string `json:"content"`
	H5          string `json:"h5"`
	GroupId     string `json:"groupId"`
}

//

type HiloUserInfo struct {
	WealthGrade uint32   `json:"wealthGrade"`
	CharmGrade  uint32   `json:"charmGrade"`
	IsVip       bool     `json:"isVip"`
	IsPretty    bool     `json:"isPretty"`
	Medals      []uint32 `json:"medals"`
	PowerName   string   `json:"powerName"` // 用户加入的国家势力的绑定群组的名称
	NobleLevel  uint16   `json:"nobleLevel"`
}

//不用返回错误
func GetHiloUserInfo(model *domain.Model, extId string) string {
	user, err := user_m.GetUserByExtId(model, extId)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%+v", extId, err)
		return ""
	}

	wealthGrade, _, err := user_m.GetWealthGrade(model, user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}
	charmGrade, _, err := user_m.GetCharmGrade(model, user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}
	isVip, _, err := user_m.IsVip(user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}
	//

	isPretty := user.IsPrettyCode()
	//
	medals, err := user_m.GetUserMedalMerge(model.Log, model.Db, user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}

	_, powerName, err := GetGroupPowerNameByUserId(model, user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}

	nobleLevel, err := noble_m.GetNobleLevel(model.Db, user.ID)
	if err != nil {
		model.Log.Errorf("extId:%v, err:%v", extId, err)
		return ""
	}

	hilo := HiloUserInfo{
		WealthGrade: wealthGrade,
		CharmGrade:  charmGrade,
		IsVip:       isVip,
		IsPretty:    isPretty,
		Medals:      medals,
		PowerName:   powerName,
		NobleLevel:  nobleLevel,
	}
	buf, err := json.Marshal(hilo)
	if err != nil {
		model.Log.Errorf("hilo:%+v, err:%v", hilo, err)
	}
	return string(buf)
}

//
var GetGroupPowerNameByUserId func(model *domain.Model, userId uint64) (uint64, string, error)

// 多协程并行发送群消息
func BroadcastGroupMessage(model *domain.Model, groupIds []string, content string, hiloUserInfo string) {
	n := config.GetGroupImConfig().MSG_PARALLEL_SIZE
	if n <= 0 {
		n = group_e.DefaultMsgParallelSize
	}
	buckets := make([][]string, n, n)
	for i := 0; i < n; i++ {
		buckets[i] = make([]string, 0)
	}
	for i := 0; i < len(groupIds); i++ {
		buckets[i%n] = append(buckets[i%n], groupIds[i])
	}
	model.Log.Infof("%d groups split into %d buckets", len(groupIds), n)

	for i, _ := range buckets {
		go func(model *domain.Model, bucketId int, groupIds []string, content string) {
			defer func() {
				if r := recover(); r != nil {
					model.Log.Errorf("BroadcastGroupMessage SYSTEM ACTION PANIC: %v, stack: %v", r, string(debug.Stack()))
				}
			}()
			for j, g := range groupIds {
				txGroupId, err := ToTxGroupId(model, g)
				if err == nil {
					model.Log.Infof("Bucket %d, sending message to groupId %d: %s", bucketId, j, txGroupId)
					tencentyun.SendCustomMsg(model.Log, txGroupId, nil, content, hiloUserInfo)
				}

				if j%10 == 0 {
					time.Sleep(time.Second)
				}
			}
		}(domain.CreateModelContext(model.MyContext), i, buckets[i], content)
	}
}
