package group_power_r

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"github.com/gin-gonic/gin"
	"github.com/jinzhu/now"
	"hilo-group/_const/enum/groupPower_e"
	"hilo-group/cv/group_power_cv"
	"hilo-group/cv/user_cv"
	"hilo-group/domain/cache/groupPower_c"
	"hilo-group/domain/model/groupPower_m"
	"hilo-group/domain/model/user_m"
	"hilo-group/domain/service/group_power_s"
	"hilo-group/myerr/bizerr"
	"hilo-group/req"
	"hilo-group/resp"
	"strconv"
	"time"
)

// @Tags 国家势力
// @Summary 家族榜单
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param period path string true "榜单周期 day:日 week:周 month:月"
// @Success 200 {object} []group_power_cv.CvGroupPowerRank
// @Router /v1/groupPower/rank/{period} [get]
func GroupPowerRank(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}
	period := c.Param("period")
	if period != "day" && period != "week" && period != "month" {
		return myContext, bizerr.InvalidParameter
	}
	var beginDate, endDate string
	switch period {
	case "day":
		beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
	case "week":
		beginDate, endDate = utils.GetMonday(time.Now()).Format("2006-01-02"), utils.GetMonday(time.Now()).AddDate(0, 0, 6).Format("2006-01-02")
	case "month":
		beginDate, endDate = now.BeginningOfMonth().Format("2006-01-02"), now.EndOfMonth().Format("2006-01-02")
	}
	var model = domain.CreateModelContext(myContext)
	response, err := group_power_s.GetGroupPowerRankResp(model, beginDate, endDate, userId, 0, 0)
	if err != nil {
		return myContext, err
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族榜单-活动
// @Success 200 {object} group_power_cv.CvGroupPowerRankMontAct
// @Router /v1/h5/groupPower/rank/month [get]
func GroupPowerMonthRankAct(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}
	var model = domain.CreateModelContext(myContext)
	// 国家区域
	var area int
	// 我的家族
	myGroupPower, err := groupPower_m.GetGroupPowerUserOrNil(model, userId)
	if err != nil {
		return myContext, err
	}
	if myGroupPower != nil && myGroupPower.GroupPowerId > 0 {
		//获取势力主
		mgrUserId, err := groupPower_m.GetGroupPowerMgr(model, myGroupPower.GroupPowerId)
		if err != nil {
			return myContext, err
		}
		_, area, err = user_m.GetUserCountryArea(model, mgrUserId)
		if err != nil {
			model.Log.Errorf("GroupPowerMonthRankAct 获取国家资源错误 userId:%d, err:%v", userId, err)
			return myContext, err
		}
	}
	if area == 0 {
		_, area, err = user_m.GetUserCountryArea(model, userId)
		if err != nil {
			model.Log.Errorf("GroupPowerMonthRankAct 获取国家资源错误 userId:%d, err:%v", userId, err)
			return myContext, err
		}
	}
	// 这个月
	beginDate, endDate := now.BeginningOfMonth().Format(utils.DATE_FORMAT), now.EndOfMonth().Format(utils.DATE_FORMAT)
	response, err := GroupPowerMonthRankActByMonth(model, userId, beginDate, endDate, time.Now().Format(utils.COMPACT_MONTH_FORMAT), area)
	if err != nil {
		return myContext, err
	}
	// 上月
	lastMonthStart := utils.GetLastMonthStart(time.Now())
	lastTime := now.New(lastMonthStart)
	beginDateLast, endDateLast := lastTime.BeginningOfMonth().Format(utils.DATE_FORMAT), lastTime.EndOfMonth().Format(utils.DATE_FORMAT)
	responseLast, err := GroupPowerMonthRankActByMonth(model, userId, beginDateLast, endDateLast, lastMonthStart.Format(utils.COMPACT_MONTH_FORMAT), area)
	if err != nil {
		return myContext, err
	}

	res := &group_power_cv.CvGroupPowerRankMontAct{
		MyGroupPower:     response.MyGroupPower,
		Items:            response.Items,
		MyGroupPowerLast: responseLast.MyGroupPower,
		ItemsLast:        responseLast.Items,
		LeftSecond:       now.EndOfMonth().Unix() - time.Now().Unix(),
	}
	resp.ResponseOk(c, res)
	return myContext, nil
}

func GroupPowerMonthRankActByMonth(model *domain.Model, userId uint64, beginDate, endDate, month string, area int) (response group_power_cv.CvGroupPowerRank, err error) {
	response, err = group_power_s.GetGroupPowerRankResp(model, beginDate, endDate, userId, groupPower_e.GroupPowerUserHas, area)
	if err != nil {
		return response, err
	}
	if response.MyGroupPower != nil {
		response.MyGroupPower.StarList, err = group_power_s.GetGroupPowerStar(model, response.MyGroupPower.Id, groupPower_e.GroupPowerStarTypeFamous, 0, 10, month)
	}
	if err != nil {
		return response, err
	}
	for i, v := range response.Items {
		response.Items[i].StarList, err = group_power_s.GetGroupPowerStar(model, v.Id, groupPower_e.GroupPowerStarTypeFamous, 0, 10, month)
		if err != nil {
			return response, err
		}
	}
	return response, nil
}

// @Tags 国家势力
// @Summary 家族日周月榜单top3
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Success 200 {object} map[string][]group_power_cv.CvGroupPowerRankData
// @Router /v1/groupPower/rankTop [get]
func GroupPowerRankTop(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	periods := []string{"day", "week", "month"}
	var response = make(map[string][]group_power_cv.CvGroupPowerRankData)
	for _, period := range periods {
		var beginDate, endDate string
		switch period {
		case "day":
			beginDate, endDate = time.Now().Format("2006-01-02"), time.Now().Format("2006-01-02")
		case "week":
			beginDate, endDate = utils.GetMonday(time.Now()).Format("2006-01-02"), utils.GetMonday(time.Now()).AddDate(0, 0, 6).Format("2006-01-02")
		case "month":
			beginDate, endDate = now.BeginningOfMonth().Format("2006-01-02"), now.EndOfMonth().Format("2006-01-02")
		}
		var model = domain.CreateModelContext(myContext)
		rank, err := groupPower_m.GetGroupPowerExpRank(model, beginDate, endDate, 3, 0, 0)
		if err != nil {
			return myContext, err
		}
		var ids []mysql.ID
		for _, g := range rank {
			ids = append(ids, g.GroupPowerId)
		}

		grades, err := groupPower_m.MGetGroupPowerGrade(model, ids)
		if err != nil {
			return myContext, err
		}
		groupPowers, err := groupPower_m.MGetGroupPowerInfoMap(model, ids)
		if err != nil {
			return myContext, err
		}
		for _, v := range rank {
			response[period] = append(response[period], group_power_cv.CvGroupPowerRankData{
				CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
					Id:        v.GroupPowerId,
					Icon:      groupPowers[v.GroupPowerId].Icon,
					Name:      groupPowers[v.GroupPowerId].Name,
					Nameplate: groupPowers[v.GroupPowerId].Nameplate,
				},
				CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
					Grade: grades[v.GroupPowerId].Grade,
					Exp:   v.Exp,
				},
				Rank: v.Rank,
			})
		}
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}

type GroupPowerStarReq struct {
	GroupPowerId mysql.ID                        `form:"groupPowerId" binding:"required"`
	Type         groupPower_e.GroupPowerStarType `form:"type" binding:"required"`
	PageSize     int                             `form:"pageSize"`
	PageIndex    int                             `form:"pageIndex"`
}

// @Tags 国家势力
// @Summary 家族之星
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupPowerId query int true "家族id"
// @Param type query integer true "类型 1:送礼 2:活跃 3:收礼物"
// @Param pageSize query int false "分页大小 默认：30" default(30)
// @Param pageIndex query int false "第几个分页，从1开始 默认：1" default(1)
// @Success 200 {object} []group_power_cv.CvGroupPowerStarData
// @Router /v1/groupPower/star [get]
func GroupPowerStar(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	param := new(GroupPowerStarReq)
	if err := c.ShouldBindQuery(param); err != nil {
		return myContext, err
	}
	if param.PageIndex <= 0 {
		param.PageIndex = 1
	}
	if param.PageSize <= 0 {
		param.PageSize = 30
	}
	offset, limit := (param.PageIndex-1)*param.PageSize, param.PageSize
	var model = domain.CreateModelContext(myContext)
	response, err := group_power_s.GetGroupPowerStar(model, param.GroupPowerId, param.Type, offset, limit, time.Now().Format(utils.COMPACT_MONTH_FORMAT))
	if err != nil {
		return myContext, err
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族之星
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param period path string true "周期 day|week|month"
// @Param groupPowerId query int true "家族id"
// @Param type query integer true "类型 1:送礼 2:活跃 3:收礼物"
// @Param pageSize query int false "分页大小 默认：30" default(30)
// @Param pageIndex query int false "第几个分页，从1开始 默认：1" default(1)
// @Success 200 {object} []group_power_cv.CvGroupPowerStarData
// @Router /v1/groupPower/star/{period} [get]
func GroupPowerStarPeriod(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	param := new(GroupPowerStarReq)
	if err := c.ShouldBindQuery(param); err != nil {
		return myContext, err
	}
	if param.PageIndex <= 0 {
		param.PageIndex = 1
	}
	if param.PageSize <= 0 {
		param.PageSize = 30
	}
	period := c.Param("period")
	if period != "day" && period != "week" && period != "month" {
		return myContext, bizerr.InvalidParameter
	}
	var model = domain.CreateModelContext(myContext)
	offset, limit := (param.PageIndex-1)*param.PageSize, param.PageSize
	rank, err := groupPower_c.GetGroupPowerStarRankPeriod(model, period, param.GroupPowerId, param.Type, offset, limit)
	if err != nil {
		return myContext, err
	}
	var response []group_power_cv.CvGroupPowerStarData
	var userIds []mysql.ID
	for _, row := range rank {
		userIds = append(userIds, row.UserId)
	}
	users, err := user_m.GetUserMapByIds(model, userIds)
	for _, row := range rank {
		user := users[row.UserId]
		score := row.Score
		if param.Type == groupPower_e.GroupPowerStarTypeActive {
			score = score / 60
		}
		if score <= 0 {
			continue
		}
		response = append(response, group_power_cv.CvGroupPowerStarData{
			User: user_cv.CvUserTiny{
				Id:         user.ID,
				ExternalId: user.ExternalId,
				Code:       user.Code,
				Nick:       user.Nick,
				Avatar:     user.Avatar,
			},
			Score: score,
		})
	}
	resp.ResponsePageBaseOk(c, response, 0, false)
	return myContext, nil
}

// @Tags 国家势力
// @Summary 家族等级页
// @Param token header string true "token"
// @Param nonce header string true "随机数字"
// @Param groupPowerId query int true "家族id"
// @Success 200 {object} group_power_cv.CvGroupPowerGradeDetail
// @Router /v1/h5/groupPower/grade/detail [get]
func GroupPowerGradeDetail(c *gin.Context) (*mycontext.MyContext, error) {
	myContext := mycontext.CreateMyContext(c.Keys)
	userId, err := req.GetUserId(c)
	if err != nil {
		return myContext, err
	}
	groupPowerId, err := strconv.ParseUint(c.Query("groupPowerId"), 10, 16)
	if err != nil || groupPowerId <= 0 {
		return myContext, bizerr.InvalidParameter
	}
	var model = domain.CreateModelContext(myContext)
	groupPowerInfo, err := groupPower_m.MGetGroupPowerInfoMap(model, []mysql.ID{groupPowerId})
	if err != nil {
		return myContext, err
	}
	gradeM, err := groupPower_m.MGetGroupPowerGrade(model, []mysql.ID{groupPowerId})
	if err != nil {
		return myContext, err
	}
	groupPower, ok := groupPowerInfo[groupPowerId]
	if !ok {
		return myContext, bizerr.GroupNotFound
	}
	grade := gradeM[groupPowerId]
	nextExp := groupPower_e.GroupPowerGradeExp[groupPower_e.GroupPowerGradeMax]
	if grade.Grade != groupPower_e.GroupPowerGradeMax {
		nextExp = groupPower_e.GroupPowerGradeExp[grade.Grade+1]
	}
	expireAt := ""
	if grade.ExpireAt.After(time.Now()) {
		expireAt = grade.ExpireAt.Add(-time.Second).Format("2006-01-02") // 减少一秒格式化天
	}
	if grade.Grade == groupPower_e.GroupPowerGrade0 {
		expireAt = utils.AddDate(now.EndOfMonth(), 0, 1).Add(-time.Second).Format("2006-01-02") // 0级别(黑铁) 没有有效期,给下个月月底
	}
	isMember := false
	if exits, id, err := groupPower_m.CheckGroupPowerUser(model, userId); err != nil {
		return myContext, err
	} else if exits && id == groupPowerId {
		isMember = true
	}
	response := group_power_cv.CvGroupPowerGradeDetail{
		GroupPowerBase: group_power_cv.CvGroupPowerBase{
			Id:        groupPower.ID,
			Icon:      groupPower.Icon,
			Name:      groupPower.Name,
			Nameplate: groupPower.Nameplate,
		},
		GroupPowerGrade: group_power_cv.CvGroupPowerGrade{
			Grade:    grade.Grade,
			Exp:      grade.Exp,
			NextExp:  nextExp,
			ExpireAt: expireAt,
		},
		ResGradeList:  group_power_cv.CvResGradeList,
		PrivilegeList: group_power_cv.GroupPowerGradePrivilegeNum[grade.Grade],
		IsMember:      isMember,
	}
	resp.ResponseOk(c, response)
	return myContext, nil
}
