package group_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"
	"hilo-group/_const/enum/group_e"
)

type GroupRoles struct {
	mysql.Entity
	UserId    uint64
	ImGroupId string
	Role      group_e.GroupRoleType
}

func CreateGroupRole(model *domain.Model, imGroupId string, userId uint64, role group_e.GroupRoleType) error {
	return model.Db.Clauses(clause.OnConflict{
		Columns:   []clause.Column{{Name: "user_id"}, {Name: "im_group_id"}},
		DoUpdates: clause.AssignmentColumns([]string{"role"}),
	}).Create(&GroupRoles{
		UserId:    userId,
		ImGroupId: imGroupId,
		Role:      role,
	}).Error
}

func RemoveGroupRole(model *domain.Model, imGroupId string, userId uint64) error {
	return model.Db.Where(&GroupRoles{ImGroupId: imGroupId, UserId: userId}).Delete(&GroupRoles{}).Error
}

// 查询用户在IM群组中的角色
func GetRoleInGroup(model *domain.Model, userId uint64, imGroupId string) (uint16, error) {
	r := GroupRoles{}
	err := model.Db.Where(&GroupRoles{
		UserId:    userId,
		ImGroupId: imGroupId}).First(&r).Error
	if err != nil {
		if err != gorm.ErrRecordNotFound {
			return 0, err
		} else {
			return 0, nil
		}
	}
	return r.Role, nil
}

// 取群中的指定角色，按时间排序
func FindRolesInGroup(db *gorm.DB, imGroupId string, role group_e.GroupRoleType) ([]uint64, error) {
	rows := make([]GroupRoles, 0)
	err := db.Model(&GroupRoles{}).Where(&GroupRoles{
		Role:      role,
		ImGroupId: imGroupId}).Order("created_time").Find(&rows).Error
	if err != nil {
		return nil, err
	}
	result := make([]uint64, 0)
	for _, i := range rows {
		result = append(result, i.UserId)
	}
	return result, nil
}

// 统计IM群组中某角色的个数
func GetRoleCountInGroup(model *domain.Model, imGroupId string, role group_e.GroupRoleType) (uint, error) {
	var count int64 = 0
	err := model.Db.Model(&GroupRoles{}).Where(&GroupRoles{
		Role:      role,
		ImGroupId: imGroupId}).Count(&count).Error
	if err != nil {
		return 0, err
	}
	return uint(count), nil
}

// 统计用户在所有群中的角色
func GetRolesByUser(model *domain.Model, userId uint64) (map[string]uint16, error) {
	data := make([]GroupRoles, 0)
	err := model.Db.Where(&GroupRoles{UserId: userId}).Find(&data).Error
	if err != nil {
		return nil, err
	}

	result := make(map[string]uint16, 0)
	for _, i := range data {
		result[i.ImGroupId] = i.Role
	}
	return result, nil
}

// 查询群组中所有有角色的成员，由级别高到低、创建时间由早到晚排列
func GetRolesInGroup(model *domain.Model, groupId string) (map[uint64]uint16, []uint64, error) {
	data := make([]GroupRoles, 0)
	err := model.Db.Where(&GroupRoles{ImGroupId: groupId}).Order("role DESC, created_time").Find(&data).Error
	if err != nil {
		return nil, nil, err
	}

	result := make(map[uint64]uint16, 0)
	orders := make([]uint64, 0)
	for _, i := range data {
		orders = append(orders, i.UserId)
		result[i.UserId] = i.Role
	}
	return result, orders, nil
}

func IsRoleGreater(model *domain.Model, groupId string, userId1, userId2 uint64) (bool, error) {
	m, _, err := GetRolesInGroup(model, groupId)
	if err != nil {
		return false, err
	}
	return m[userId1] > m[userId2], nil
}

func GetGroupOwner(model *domain.Model, groupId string) (uint64, error) {
	gr := GroupRoles{ImGroupId: groupId, Role: group_e.GROUP_OWNER}
	err := model.Db.Where(&gr).First(&gr).Error
	if err != nil && err != gorm.ErrRecordNotFound {
		return 0, err
	}
	return gr.UserId, nil
}

func GetGroups(db *gorm.DB, userId uint64, role group_e.GroupRoleType) ([]string, error) {
	gr := GroupRoles{UserId: userId, Role: role}
	rows := make([]GroupRoles, 0)
	err := db.Where(&gr).Order("created_time").Find(&rows).Error
	if err != nil {
		return nil, err
	}
	result := make([]string, 0)
	for _, i := range rows {
		result = append(result, i.ImGroupId)
	}
	return result, nil
}
