package res_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"gorm.io/gorm"
	"hilo-group/_const/enum/res_e"
	"hilo-group/myerr"
)

type ResCountry struct {
	mysql.Entity
	*domain.Model     `gorm:"-"`
	Name              mysql.Str
	ShortName         mysql.Str
	Code              mysql.Str
	Icon              mysql.Str
	Icon2             mysql.Str
	Lang              mysql.Str
	IsCommon          mysql.UserYesNo
	IsDefault         mysql.UserYesNo
	Status            mysql.UserYesNo
	StandardShortName mysql.Str
	AreaCode          mysql.Str
	AreaCodeName      mysql.Str
	Area              mysql.Str
}

type ResLanguage struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	Name          mysql.Str
	Remark        mysql.Str
	ResCountryId  mysql.ID
}

type ResMsgTranslate struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	Language      mysql.Str
	Title         mysql.Str
	Content       mysql.Str
	IconUrl       mysql.Str
	MsgType       mysql.Type
	Type          uint32
	//跳转类型 0:无跳转 1：网页跳转， 2：跳转到用户 3：跳转到群组
	ActionType mysql.Type
	ActionUrl  mysql.Str
}

// 复制消息并替换图片和URL
func (rmt *ResMsgTranslate) CopyRecord(db *gorm.DB, newType uint64, iconUrl, actionUrl string) error {
	return db.Exec("REPLACE INTO res_msg_translate (language,title,content,?,msg_type,?,action_type,?) "+
		"SELECT language,title,content,icon_url,msg_type,10001,action_type,action_url FROM res_msg_translate "+
		"WHERE msg_type = ? and type = 39", iconUrl, newType, actionUrl, rmt.MsgType, rmt.Type).Error
}

//获取默认的国家
func GetCountryDefault(model *domain.Model) (*ResCountry, error) {
	var resCountry ResCountry
	if err := model.Db.Where(&ResCountry{
		IsDefault: mysql.USER,
	}).First(&resCountry).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	return &resCountry, nil
}

// 获取所有国家的icon信息
func GetAllCountries(model *domain.Model) (map[string]string, error) {
	var countrys []ResCountry
	if err := model.Db.Model(&ResCountry{}).Where(&ResCountry{
		Status: mysql.USER,
	}).Find(&countrys).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}

	result := make(map[string]string, 0)
	for _, i := range countrys {
		result[i.Name] = i.Icon
	}
	return result, nil
}

// 获取所有国家的Area信息
func GetAllCountriesArea(model *domain.Model) (map[string]string, error) {
	var countrys []ResCountry
	if err := model.Db.Model(&ResCountry{}).Where(&ResCountry{
		Status: mysql.USER,
	}).Find(&countrys).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}

	result := make(map[string]string, 0)
	for _, i := range countrys {
		result[i.Name] = i.Area
	}
	return result, nil
}

func GetAllCountryByFilter(model *domain.Model, shortNames []string) ([]ResCountry, error) {
	var countrys []ResCountry
	if err := model.Db.Model(&ResCountry{}).Where(&ResCountry{
		Status: mysql.USER,
	}).Find(&countrys).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	countryMap := map[string]ResCountry{}

	//转成map
	for i, _ := range countrys {
		countryMap[countrys[i].ShortName] = countrys[i]
	}

	results := make([]ResCountry, 0, len(shortNames))
	for i, _ := range shortNames {
		if country, flag := countryMap[shortNames[i]]; flag {
			results = append(results, country)
		}
	}
	return results, nil
}

//通过standardShortName获取国家
func GetCountryByStandardShortName(model *domain.Model, standardShortName mysql.Str) (*ResCountry, error) {
	var resCountry ResCountry
	if err := model.Db.Where(&ResCountry{
		StandardShortName: standardShortName,
		Status:            mysql.USER,
	}).First(&resCountry).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		} else {
			return nil, myerr.WrapErr(err)
		}
	}
	resCountry.Model = model
	return &resCountry, nil
}

//通过shortName获取国家
func GetCountryByShortName(model *domain.Model, shortName mysql.Str) (*ResCountry, error) {
	var resCountry ResCountry
	err := model.Db.Where(&ResCountry{
		ShortName: shortName,
	}).First(&resCountry).Error
	return &resCountry, myerr.WrapErr(err)
}

//通过name获取国家
func GetCountryByName(model *domain.Model, name mysql.Str) (*ResCountry, error) {
	var resCountry ResCountry
	err := model.Db.WithContext(model.Context).Where(&ResCountry{
		Name: name,
	}).First(&resCountry).Error
	return &resCountry, myerr.WrapErr(err)
}

//通过code获取国家
func GetCountryByCode(model *domain.Model, code mysql.Str) (*ResCountry, error) {
	if code == "" {
		return nil, nil
	}
	var resCountry ResCountry
	err := model.Db.Where(&ResCountry{
		Code: code,
	}).First(&resCountry).Error
	if err == nil {
		return &resCountry, myerr.WrapErr(err)
	} else if err == gorm.ErrRecordNotFound {
		return nil, nil
	} else {
		return nil, myerr.WrapErr(err)
	}
}

//通过id获取国家
func GetCountryById(model *domain.Model, id mysql.ID) (*ResCountry, error) {
	var resCountry ResCountry
	err := model.Db.First(&resCountry, id).Error
	return &resCountry, myerr.WrapErr(err)
}

//通过语言，找到国家
func GetCountryByLanguage(model *domain.Model, name mysql.Str) (*ResCountry, error) {
	if name == "" {
		return nil, nil
	}
	var language ResLanguage
	err := model.Db.Where(&ResLanguage{
		Name: name,
	}).First(&language).Error
	if err == nil {
		return GetCountryById(model, language.ResCountryId)
	} else if err == gorm.ErrRecordNotFound {
		return nil, nil
	} else {
		return nil, myerr.WrapErr(err)
	}
}

//通过国家,找到对应的语言
func GetLangeByCountry(db *gorm.DB, country mysql.Str) (string, error) {
	var r ResCountry
	err := db.Where(&ResCountry{
		Name: country,
	}).First(&r).Error
	if err == nil {
		return r.Lang, nil
	} else if err == gorm.ErrRecordNotFound {
		return res_e.DEFAULT_LANG, nil
	} else {
		return "", myerr.WrapErr(err)
	}
}

//获取所有国家名字列表
func GetCountryNameList(model *domain.Model) ([]string, error) {
	res := make([]string, 0)
	err := model.DB().Table("res_country").Select("distinct(name) name").Pluck("name", &res).Error
	return res, myerr.WrapErr(err)
}
