package fruit_m

import (
	"git.hilo.cn/hilo-common/resource/mysql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"
	"hilo-group/_const/enum/fruitMachine_e"
	"hilo-group/myerr"
)

type FruitMachinePlayer struct {
	mysql.Entity
	Date     string
	Round    uint
	UserId   uint64
	FruitNum uint
	Award    uint
}

func (fmp *FruitMachinePlayer) Create(db *gorm.DB) (int64, error) {
	result := db.Clauses(clause.OnConflict{DoNothing: true}).
		Create(fmp)
	return result.RowsAffected, result.Error
}

func (fmp *FruitMachinePlayer) InsertUpdateFruitNum(db *gorm.DB) (int64, error) {
	result := db.Clauses(clause.OnConflict{
		DoUpdates: clause.Assignments(map[string]interface{}{
			"fruit_num": gorm.Expr("fruit_num + IF(fruit_num>=?, 0, ?)", fruitMachine_e.MaxFruitStakeCount, fmp.FruitNum)}),
	}).Create(fmp)
	return result.RowsAffected, result.Error
}

func (fmp *FruitMachinePlayer) GetWinners(db *gorm.DB) ([]FruitMachinePlayer, error) {
	result := make([]FruitMachinePlayer, 0)
	if err := db.Where(fmp).Where("award > 0").Order("award DESC").Find(&result).Error; err != nil {
		return nil, err
	}
	return result, nil
}

func (fmp *FruitMachinePlayer) UpdateAward(db *gorm.DB, award uint) (int64, error) {
	result := db.Model(&FruitMachinePlayer{}).Where(fmp).Update("award", award)
	return result.RowsAffected, result.Error
}

func (fmp *FruitMachinePlayer) Find(db *gorm.DB) ([]FruitMachinePlayer, error) {
	rows := make([]FruitMachinePlayer, 0)
	if err := db.Where(fmp).Order("`date` DESC, round DESC").Find(&rows).Error; err != nil {
		return nil, err
	}
	return rows, nil
}

func SumAward(db *gorm.DB, beginDate, endDate string) (map[uint64]uint, error) {
	type sum struct {
		UserId uint64
		S      uint
	}
	rows := make([]sum, 0)
	if err := db.Model(&FruitMachinePlayer{}).Select("user_id,SUM(award) AS s").Where("date BETWEEN ? AND ?", beginDate, endDate).
		Group("user_id").Find(&rows).Error; err != nil {
		return nil, err
	}
	result := make(map[uint64]uint, 0)
	for _, i := range rows {
		result[i.UserId] = i.S
	}
	return result, nil
}

func SumAwardAll(db *gorm.DB, userId uint64) (uint64, error) {
	type summary struct {
		Sum uint64
	}
	results := []summary{}
	err := db.Model(&FruitMachinePlayer{}).
		Select("SUM(award) AS sum").
		Where(&FruitMachinePlayer{
			UserId: userId,
		}).Find(&results).Error
	if err != nil {
		return 0, err
	}
	if len(results) == 0 {
		return 0, nil
	}
	return results[0].Sum, err
}

func MaxAward(db *gorm.DB, userId uint64) (uint, error) {
	fruitMachinePlayers := []FruitMachinePlayer{}
	if err := db.Model(&FruitMachinePlayer{}).Where(&FruitMachinePlayer{
		UserId: userId,
	}).Order("award desc").Find(&fruitMachinePlayers).Error; err != nil {
		return 0, myerr.WrapErr(err)
	}
	if len(fruitMachinePlayers) == 0 {
		return 0, nil
	}
	return fruitMachinePlayers[0].Award, nil
}

type RangeSum struct {
	UserId  uint64
	Diamond uint32
}

func SumAwardByRange(db *gorm.DB, beginDate, endDate string) ([]RangeSum, error) {
	rows := make([]RangeSum, 0)
	if err := db.Model(&FruitMachinePlayer{}).Select("user_id, SUM(award) AS diamond").
		Where("date BETWEEN ? AND ?", beginDate, endDate).
		Group("user_id").Having("diamond > 0").Order("diamond DESC").Find(&rows).Error; err != nil {
		return nil, err
	}
	return rows, nil
}

func SumUserAward(db *gorm.DB, userId uint64, beginDate, endDate string) (uint32, error) {
	type sum struct {
		UserId uint64
		S      uint32
	}
	rows := make([]sum, 0)
	if err := db.Model(&FruitMachinePlayer{}).Select("user_id,SUM(award) AS s").
		Where("user_id = ? AND date BETWEEN ? AND ?", userId, beginDate, endDate).
		Group("user_id").Find(&rows).Error; err != nil {
		return 0, err
	}
	if len(rows) <= 0 {
		return 0, nil
	}
	return rows[0].S, nil
}
