package group_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/config"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/utils"
	"gorm.io/gorm"
	"hilo-group/_const/enum/group_e"
	"hilo-group/myerr"
	"hilo-group/myerr/bizerr"
	"time"
)

type GroupCustomTheme struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	ImGroupId     string
	PicUrl        mysql.Str
	ExpireTime    time.Time
	Using         mysql.YesNo
}

func GetGroupCustomThemeById(model *domain.Model, imGroupId string, id mysql.ID) (*GroupCustomTheme, error) {
	groupCustomTheme := GroupCustomTheme{}
	if err := model.Db.Model(&GroupCustomTheme{
		ImGroupId: imGroupId,
	}).First(&groupCustomTheme, id).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	groupCustomTheme.Model = model
	return &groupCustomTheme, nil
}

func (groupCustomTheme *GroupCustomTheme) SetUsing(userId mysql.ID) (*GroupCustomTheme, error) {
	// 判断有没有权限修改资料
	role, err := GetRoleInGroup(groupCustomTheme.Model, userId, groupCustomTheme.ImGroupId)
	if err != nil {
		return nil, err
	}
	if role != group_e.GROUP_OWNER && role != group_e.GROUP_MANAGER {
		//return nil, myerr.NewSysError("No privileges")
		return nil, bizerr.NoPrivileges
	}

	//将其它的赋值为未使用
	if err := groupCustomTheme.Db.Model(&GroupCustomTheme{}).Where(&GroupCustomTheme{ImGroupId: groupCustomTheme.ImGroupId}).UpdateColumn("using", mysql.NO).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	groupCustomTheme.Using = mysql.YES
	return groupCustomTheme, nil
}

func AddGroupCustomTheme(model *domain.Model, userId mysql.ID, imGroupId string, picUrl string) (*GroupCustomTheme, error) {
	// 判断有没有权限修改资料
	role, err := GetRoleInGroup(model, userId, imGroupId)
	if err != nil {
		return nil, err
	}
	if role != group_e.GROUP_OWNER && role != group_e.GROUP_MANAGER {
		//return nil, myerr.NewSysError("No privileges")
		return nil, bizerr.NoPrivileges
	}

	var n int64 = 0
	if err := model.Db.Model(&GroupCustomTheme{}).Where(&GroupCustomTheme{ImGroupId: imGroupId}).Where("expire_time > ?", time.Now()).Count(&n).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	if n >= int64(config.GetGroupCustomThemeConfig().PIC_LIMIT) {
		return nil, bizerr.GroupCustomThemeLimit
	}
	//校验权限，产品没说，应该有

	//将其它的赋值为未使用
	if err := model.Db.Model(&GroupCustomTheme{}).Where(&GroupCustomTheme{ImGroupId: imGroupId}).UpdateColumn("using", mysql.NO).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	picUrl = utils.MakeFullUrl(picUrl)

	return &GroupCustomTheme{
		Model:      model,
		ImGroupId:  imGroupId,
		PicUrl:     picUrl,
		ExpireTime: time.Now().AddDate(0, 0, config.GetGroupCustomThemeConfig().DAY),
		Using:      mysql.YES,
	}, nil
}

//获取最新展示的主题
func GetShowCustomTheme(model *domain.Model, imGroupId string) (uint64, string, error) {
	groupCustomTheme := GroupCustomTheme{}
	if err := model.Db.Model(&GroupCustomTheme{}).Where(&GroupCustomTheme{ImGroupId: imGroupId}).Where("expire_time > ?", time.Now()).Order("`using` asc, expire_time asc").First(&groupCustomTheme).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return 0, "", nil
		} else {
			return 0, "", myerr.WrapErr(err)
		}
	} else {
		return groupCustomTheme.ID, groupCustomTheme.PicUrl, nil
	}
}
