package user_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"hilo-group/_const/enum/user_e"
	"hilo-group/myerr"
	"hilo-group/myerr/bizerr"
)

//用户信息
type User struct {
	mysql.Entity
	*domain.Model `gorm:"-"`
	ExternalId    mysql.Str
	Avatar        mysql.Str
	DefaultAvatar bool
	Nick          mysql.Str
	Sex           mysql.Sex
	Birthday      mysql.Timestamp
	Country       mysql.Str
	CountryIcon   mysql.Str
	Language      mysql.Str
	Description   mysql.Str
	Code          mysql.Str
	OriginCode    mysql.Str
	IsPush        mysql.OpenClose
	IsShowAge     mysql.OpenClose
	Status        user_e.UserStatus
	DeviceType    mysql.Str
	LogoutTime    int64
}

type UserTiny struct {
	ID           uint64 `json:"id,omitempty"`
	ExternalId   string `json:"externalId"`
	Avatar       string `json:"avatar"`
	Nick         string `json:"nick"`
	Sex          uint8  `json:"sex"` // 1男2女
	Code         string `json:"code"`
	Description  string `json:"description"`
	Country      string `json:"country"`
	CountryIcon  string `json:"countryIcon"`
	IsPrettyCode bool   `json:"isPrettyCode"` // 是否靓号
	IsLogout     bool   `json:"isLogout"`     //是否注销 true:已经注销， false:没有注销
	//生日，如果是其它人用户信息，年龄则按照是否展示显示，如果是本人，年龄则按照是否存在展示
	Birthday *uint64 `json:"birthday"`
}

func (u User) IsPrettyCode() bool {
	return u.Code != u.OriginCode
}

//获取用户
func GetUser(model *domain.Model, id mysql.ID) (*User, error) {
	var user User
	if err := model.Db.WithContext(model.Context).Where(&User{
		Entity: mysql.Entity{ID: id},
	}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

// 通过code获取用户
func GetUserByCode(model *domain.Model, code string) (*User, error) {
	if code == "" {
		return nil, bizerr.InvalidParameter
	}
	var user User
	if err := model.Db.Where(&User{Code: code}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

//获取用户信息
func GetUsersByIds(model *domain.Model, userIds []mysql.ID) ([]User, error) {
	var users []User
	if err := model.Db.Model(User{}).Where("id IN (?)", userIds).Find(&users).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	return users, nil
}

// 分批获取userIds
func GetUserMapByIds(model *domain.Model, userIds []mysql.ID) (map[mysql.ID]User, error) {
	result := make(map[mysql.ID]User, 0)
	end := 500
	if end > len(userIds) {
		end = len(userIds)
	}
	var rows []User
	start := 0
	for start < len(userIds) {
		tmp, err := GetUsersByIds(model, userIds[start:end])
		if err != nil {
			return result, err
		} else {
			rows = append(rows, tmp...)
		}
		start += 500
		end += 500
		if end > len(userIds) {
			end = len(userIds)
		}
		//model.Log.Infof("GetUserMapByIds start:%v-end:%v", start, end)
	}
	for _, i := range rows {
		result[i.ID] = i
	}
	return result, nil
}

// 通过externalId获取用户
func GetUserByExtId(model *domain.Model, externalId string) (*User, error) {
	var user User
	if err := model.Db.Where(&User{ExternalId: externalId}).First(&user).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	user.Model = model
	return &user, nil
}

// 通过externalId批量获取用户
func BatchGetUserByExtIds(model *domain.Model, extIds []string) ([]User, error) {
	users := make([]User, 0)
	if err := model.Db.Where("external_id IN ?", extIds).Find(&users).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	for _, i := range users {
		i.Model = model
	}
	return users, nil
}

//获取用户
func GetUsers(model *domain.Model, ids []mysql.ID) ([]*User, error) {
	res := make([]*User, 0)
	if err := model.Db.WithContext(model.Context).Where("id in (?)", ids).Find(&res).Error; err != nil {
		return nil, myerr.WrapErr(err)
	}
	return res, nil
}
