package group_c

import (
	"context"
	"encoding/json"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/resource/redisCli"
	"github.com/go-redis/redis/v8"
	"hilo-group/_const/redis_key"
	"hilo-group/_const/redis_key/group_k"
	"hilo-group/domain/model/group_m"
	"strings"
	"time"
)

func init() {
	group_m.FuncAddEditGroupCd = AddEditGroupCd
}

func GetGroupMemberCount(groupId string) (int, error) {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Get(context.Background(), key).Int()
}

func SetGroupMemberCount(groupId string, count uint, ttl time.Duration) error {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Set(context.Background(), key, count, ttl).Err()
}

func ClearGroupMemberCount(groupId string) error {
	key := redis_key.GetGroupMemCountKey(groupId)
	return redisCli.RedisClient.Del(context.Background(), key).Err()
}

func getGroupMemberKey(groupId string) string {
	return strings.Replace(redis_key.GroupMemberPrefix, "{groupId}", groupId, -1)
}

func getGroupConsumeKey(groupId string) string {
	return strings.Replace(redis_key.GroupConsumePrefix, "{groupId}", groupId, -1)
}

func GetGroupMember(groupId string) ([]string, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SMembers(context.Background(), key).Result()
}

func GetGroupMemberCard(groupId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SCard(context.Background(), key).Result()
}

func GetGroupMemberAsMap(groupId string) (map[string]struct{}, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SMembersMap(context.Background(), key).Result()
}

func SetExists(groupId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.Exists(context.Background(), key).Result()
}

func addGroupMember(groupId string, extIds []string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SAdd(context.Background(), key, extIds).Result()
}

func AddGroupMember(groupId string, extIds []string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SAdd(context.Background(), key, extIds).Result()
}

func RemoveGroupMember(groupId string, externalId string) (int64, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.SRem(context.Background(), key, externalId).Result()
}

func SetGroupMemberTTL(groupId string, ttl time.Duration) (bool, error) {
	key := getGroupMemberKey(groupId)
	return redisCli.RedisClient.Expire(context.Background(), key, ttl).Result()
}

func GetGroupConsume(groupId string) (uint64, error) {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Get(context.Background(), key).Uint64()
}

func SetGroupConsume(groupId string, consume uint64, ttl time.Duration) error {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Set(context.Background(), key, consume, ttl).Err()
}

func ClearGroupConsume(groupId string) error {
	key := getGroupConsumeKey(groupId)
	return redisCli.RedisClient.Del(context.Background(), key).Err()
}

// 增加编辑用户cd
// 6小时cd
func AddEditGroupCd(model *domain.Model, imGroupId mysql.Str) error {
	key := group_k.GetEditGroupCDKey(imGroupId)
	return model.Redis.SetEX(model, key, 1, time.Hour*6).Err()
}

// 是否编辑群组cd中
func IsEditGroupCd(model *domain.Model, imGroupId mysql.Str) bool {
	key := group_k.GetEditGroupCDKey(imGroupId)
	n, err := model.Redis.Exists(model, key).Result()
	if err != nil {
		model.Log.Errorf("IsEditGroupCd fail:%v", err)
	}
	return n > 0
}

//
func GetGroupInfoCache(model *domain.Model, imGroupId mysql.Str) (*group_m.GroupInfo, error) {
	key := group_k.GetGroupInfoKey(imGroupId)
	data, err := model.Redis.Get(model, key).Bytes()
	if err != nil {
		if err == redis.Nil {
			return nil, nil
		}
		model.Log.Errorf("GetGroupInfo fail,imGroupId:%v,err:%v", imGroupId, err)
		return nil, err
	}
	res := new(group_m.GroupInfo)
	err = json.Unmarshal(data, &res)
	if err != nil {
		model.Log.Errorf("GetGroupInfo fail,imGroupId:%v,err:%v", imGroupId, err)
		return nil, err
	}
	return res, nil
}

//
func SetGroupInfoCache(model *domain.Model, info *group_m.GroupInfo) error {
	key := group_k.GetGroupInfoKey(info.ImGroupId)
	data, err := json.Marshal(info)
	if err != nil {
		model.Log.Errorf("SetGroupInfo fail,info:%+v,err:%v", info, err)
		return err
	}
	err = model.Redis.Set(model, key, data, time.Minute*5).Err()
	if err != nil {
		model.Log.Errorf("SetGroupInfo fail,info:%+v,err:%v", info, err)
		return err
	}
	return nil
}

// 增加领取群组扶持ip次数
func IncrGroupSupportAwardIp(model *domain.Model, ip mysql.Str) (times int64, err error) {
	key := group_k.GetGroupSupportAwardIpKey(ip)
	times, err = model.Redis.Incr(model, key).Result()
	if err != nil {
		model.Log.Errorf("IncrGroupSupportAwardIp fail,ip:%v,err:%v", ip, err)
	} else {
		model.Redis.Expire(model, key, time.Hour*24*6) // 1周1次,ttl=6天吧
	}
	return
}

// 获取领取群组扶持ip次数
func GetGroupSupportAwardIpTimes(model *domain.Model, ip mysql.Str) (times int64, err error) {
	key := group_k.GetGroupSupportAwardIpKey(ip)
	times, err = model.Redis.Get(model, key).Int64()
	if err != nil && err != redis.Nil {
		model.Log.Errorf("GetGroupSupportAwardIpTimes fail,ip:%v,err:%v", ip, err)
		return
	}
	return times, nil
}
