package diamond_m

import (
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mylogrus"
	"git.hilo.cn/hilo-common/resource/mysql"
	"gorm.io/gorm"
	"hilo-group/common"
	"hilo-group/myerr"
)

type DiamondDealer struct {
	mysql.Entity
	UserId        uint64
	Diamond       uint32
	Wechat        string
	Whatsapp      string
	Contact       string
	Status        int
	HasInvite     int
	IsFamilyAgent int8
}

func (dealer *DiamondDealer) Save(db *gorm.DB) error {
	if dealer.ID > 0 {
		db = db.Omit("diamond", "status")
	} else {
		dealer.Status = common.SWITCH_ON
	}
	return db.Save(dealer).Error
}

func (dealer *DiamondDealer) Get(db *gorm.DB) error {
	return db.Where(dealer).First(dealer).Error
}

func IsDiamondDealer(db *gorm.DB, userId uint64) (bool, error) {
	dd := DiamondDealer{UserId: userId, Status: common.SWITCH_ON}
	if err := dd.Get(db); err != nil {
		if err == gorm.ErrRecordNotFound {
			return false, nil
		} else {
			return false, err
		}
	}
	return true, nil
}

// 是否是家族代理
func IsFamilyDiamondDealer(db *gorm.DB, userId uint64) (bool, error) {
	dd := DiamondDealer{UserId: userId, Status: common.SWITCH_ON, IsFamilyAgent: 1}
	if err := dd.Get(db); err != nil {
		if err == gorm.ErrRecordNotFound {
			return false, nil
		} else {
			return false, err
		}
	}
	return true, nil
}

func GetDiamondDealer(db *gorm.DB, userId uint64) (*DiamondDealer, error) {
	dd := &DiamondDealer{UserId: userId, Status: common.SWITCH_ON}
	if err := dd.Get(db); err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		} else {
			return nil, myerr.WrapErr(err)
		}
	}
	return dd, nil
}

func (dealer *DiamondDealer) GetAll(db *gorm.DB, status *uint8) ([]DiamondDealer, error) {
	rows := make([]DiamondDealer, 0)
	if status == nil {
		if err := db.Where(dealer).Find(&rows).Error; err != nil {
			return nil, err
		}
	} else if err := db.Where(dealer).Where("status = ?", *status).Find(&rows).Error; err != nil {
		return nil, err
	}
	return rows, nil
}

func (dealer *DiamondDealer) Remove(db *gorm.DB) (int64, error) {
	result := db.Where(dealer).Delete(&DiamondDealer{})
	return result.RowsAffected, result.Error
}

func (dealer *DiamondDealer) SafeReduceDiamond(db *gorm.DB, diamond uint32) (int64, error) {
	result := db.Model(&DiamondDealer{}).Where(dealer).Where("diamond >= ?", diamond).UpdateColumn("diamond", gorm.Expr("diamond - ?", diamond))
	return result.RowsAffected, result.Error
}

func (dealer *DiamondDealer) AddDiamond(db *gorm.DB, diamond uint32) (int64, error) {
	result := db.Model(&DiamondDealer{}).Where(dealer).Where("status = ?", common.SWITCH_ON).UpdateColumn("diamond", gorm.Expr("diamond + ?", diamond))
	return result.RowsAffected, result.Error
}

func (dealer *DiamondDealer) SetStatus(db *gorm.DB, status uint8) error {
	return db.Model(&DiamondDealer{}).Where(dealer).Update("status", status).Error
}

type DealerTransferDetail struct {
	mysql.Entity
	DealerId   uint64
	ReceiverId uint64
	Diamond    uint32
	Dollar     uint
}

func (dtd *DealerTransferDetail) Create(db *gorm.DB) error {
	return db.Create(dtd).Error
}

func (dfd *DealerTransferDetail) Find(db *gorm.DB, offset, limit int) ([]DealerTransferDetail, error) {
	rows := make([]DealerTransferDetail, 0)
	if err := db.Where(dfd).Order("created_time DESC").Offset(offset).Limit(limit).Find(&rows).Error; err != nil {
		return nil, err
	}
	return rows, nil
}

func (dfd *DealerTransferDetail) CountByDealer(db *gorm.DB, dealerIds []uint64) (map[uint64]uint, error) {
	type summary struct {
		DealerId uint64
		C        uint
	}
	rows := make([]summary, 0)
	if err := db.Model(&DealerTransferDetail{}).Where("dealer_id IN ?", dealerIds).
		Select("dealer_id, COUNT(0) AS c").Group("dealer_id").Find(&rows).Error; err != nil {
		return nil, err
	}
	result := make(map[uint64]uint, 0)
	for _, i := range rows {
		result[i.DealerId] = i.C
	}
	return result, nil
}

func (dfd *DealerTransferDetail) Sum(db *gorm.DB) (uint, uint32, uint, error) {
	type summary struct {
		C       uint
		Diamond uint32
		Dollar  uint
	}
	s := summary{}
	if err := db.Model(&DealerTransferDetail{}).Where(dfd).
		Select("COUNT(0) AS c, SUM(diamond) AS diamond, SUM(dollar) AS dollar").First(&s).Error; err != nil {
		return 0, 0, 0, err
	}
	return s.C, s.Diamond, s.Dollar, nil
}

type DealerChargeDetail struct {
	mysql.Entity
	DealerId      uint64
	MgrId         uint64
	Diamond       uint32
	Dollar        uint
	PaymentMethod string
	Type          uint8
	Money         float32 `json:"money"`    // 支付的货币数值
	Currency      string  `json:"currency"` // 支付货币
	GoodsId       string  `json:"goodsId"`  // 充值钻石套餐id/商品id
}

func (dcd *DealerChargeDetail) Create(db *gorm.DB) error {
	return db.Create(dcd).Error
}

type DealerCountry struct {
	DealerId uint64
	Country  string
}

func (dc *DealerCountry) Find(db *gorm.DB) ([]DealerCountry, error) {
	rows := make([]DealerCountry, 0)
	if err := db.Where(dc).Find(&rows).Error; err != nil {
		return nil, err
	}
	return rows, nil
}

func (dc *DealerCountry) Delete(db *gorm.DB) error {
	return db.Where(dc).Delete(&DealerCountry{}).Error
}

func (dc *DealerCountry) BatchInsert(db *gorm.DB, dealerId uint64, countries []string) error {
	rows := make([]DealerCountry, 0)
	for _, i := range countries {
		rows = append(rows, DealerCountry{
			DealerId: dealerId,
			Country:  i,
		})
	}
	return db.Create(&rows).Error
}

// 活动排行榜黑名单 获取所有已经上架代理币商的群组id
func GetAllValidActivityBlackImGroupIds(model *domain.Model) []mysql.Str {
	//return []mysql.Str{"HTGS#a88745892", "@TGS#3UW6RFSIX"}
	var imGroupIds []mysql.Str
	subQuery := model.Db.WithContext(model).Model(DiamondDealer{}).Where("status = 1").Select("user_id")
	if err := model.Db.WithContext(model).Table("group_info").Select("im_group_id").Where("owner in (?)", subQuery).Scan(&imGroupIds).Error; err != nil {
		mylogrus.MyLog.Errorf("GetAllValidActivityBlackImGroupIds fail:%v", err)
	}
	imGroupIds = append(imGroupIds, "HTGS#a88745892", "@TGS#3UW6RFSIX")
	return imGroupIds
}
