package group_power_s

import (
	"encoding/json"
	"git.hilo.cn/hilo-common/_const/enum/diamond_e"
	"git.hilo.cn/hilo-common/domain"
	"git.hilo.cn/hilo-common/mycontext"
	"git.hilo.cn/hilo-common/resource/mysql"
	"git.hilo.cn/hilo-common/txop/award_tx"
	"git.hilo.cn/hilo-common/utils"
	"github.com/jinzhu/now"
	"hilo-group/_const/enum/groupPower_e"
	"hilo-group/_const/enum/msg_e"
	"hilo-group/cv/group_power_cv"
	"hilo-group/cv/user_cv"
	"hilo-group/domain/cache/user_c"
	"hilo-group/domain/event/group_power_ev"
	"hilo-group/domain/model/groupPower_m"
	"time"
	"hilo-group/domain/model/msg_m"
	"hilo-group/domain/model/noble_m"
	"hilo-group/domain/model/user_m"
	"hilo-group/myerr/bizerr"
)

type GroupPowerService struct {
	svc *domain.Service
}

func NewGroupPowerService(myContext *mycontext.MyContext) *GroupPowerService {
	svc := domain.CreateService(myContext)
	return &GroupPowerService{svc}
}

//用户加入国家势力
func (s *GroupPowerService) GroupPowerUserJoin(groupPowerId mysql.ID, userId mysql.ID) error {
	return s.svc.Transactional(func() error {
		model := domain.CreateModel(s.svc.CtxAndDb)
		groupPower, err := groupPower_m.GetGroupPower(model, groupPowerId)
		if err != nil {
			return err
		}
		groupPowerUser, err := groupPower.UserJoin(userId)
		if err != nil {
			return err
		}
		if err := groupPowerUser.Persistent(); err != nil {
			return err
		}
		return group_power_ev.PublishGroupPowerJoin(model, &group_power_ev.GroupPowerJoinEvent{
			UserId:       userId,
			GroupPowerId: groupPowerId,
		})
	})
}

//用户退出国家势力
func (s *GroupPowerService) GroupPowerUserLeave(groupPowerId mysql.ID, userId mysql.ID) (int, error) {
	remainSeconds := 0
	err := s.svc.Transactional(func() error {
		model := domain.CreateModel(s.svc.CtxAndDb)
		groupPower, err := groupPower_m.GetGroupPower(model, groupPowerId)
		if err != nil {
			return err
		}
		groupPowerUser, timeDiff, err := groupPower.UserLeave(userId)
		if err != nil {
			if timeDiff > 0 {
				remainSeconds = int(timeDiff.Seconds())
				return err
			}
			return err
		}
		if err := groupPowerUser.Persistent(); err != nil {
			return err
		}
		return group_power_ev.PublishGroupPowerLeave(model, &group_power_ev.GroupPowerLeaveEvent{
			UserId:       userId,
			GroupPowerId: groupPowerId,
		})
	})
	return remainSeconds, err
}

func GetGroupPowerRankResp(model *domain.Model, beginDate, endDate string, userId uint64, gpStatus groupPower_e.GroupPowerStatus,
	area int) (response group_power_cv.CvGroupPowerRank, err error) {
	rank, err := groupPower_m.GetGroupPowerExpRank(model, beginDate, endDate, 30, gpStatus, area)
	if err != nil {
		return
	}
	var ids []mysql.ID
	for _, g := range rank {
		ids = append(ids, g.GroupPowerId)
	}
	var myGroupPower *groupPower_m.GroupPowerUser
	if userId > 0 {
		myGroupPower, err = groupPower_m.GetGroupPowerUserOrNil(model, userId)
		if err != nil {
			return
		}
	}
	if myGroupPower != nil {
		ids = append(ids, myGroupPower.GroupPowerId)
	}
	grades, err := groupPower_m.MGetGroupPowerGrade(model, ids)
	if err != nil {
		return
	}
	groupPowers, err := groupPower_m.MGetGroupPowerInfoMap(model, ids)
	if err != nil {
		return
	}
	if myGroupPower != nil {
		myGroupPowerRank, err := groupPower_m.GetMyGroupPowerExpRank(model, beginDate, endDate, myGroupPower.GroupPowerId)
		if err != nil {
			return response, err
		}
		response.MyGroupPower = &group_power_cv.CvGroupPowerRankData{
			CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
				Id:        myGroupPower.GroupPowerId,
				Icon:      groupPowers[myGroupPower.GroupPowerId].Icon,
				Name:      groupPowers[myGroupPower.GroupPowerId].Name,
				Nameplate: groupPowers[myGroupPower.GroupPowerId].Nameplate,
			},
			CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
				Grade: grades[myGroupPower.GroupPowerId].Grade,
				Exp:   myGroupPowerRank.Exp,
			},
			Rank: 0,
		}
	}
	for _, v := range rank {
		if response.MyGroupPower != nil && v.GroupPowerId == response.MyGroupPower.Id {
			response.MyGroupPower.Rank = v.Rank
		}
		response.Items = append(response.Items, group_power_cv.CvGroupPowerRankData{
			CvGroupPowerBase: group_power_cv.CvGroupPowerBase{
				Id:        v.GroupPowerId,
				Icon:      groupPowers[v.GroupPowerId].Icon,
				Name:      groupPowers[v.GroupPowerId].Name,
				Nameplate: groupPowers[v.GroupPowerId].Nameplate,
			},
			CvGroupPowerGrade: group_power_cv.CvGroupPowerGrade{
				Grade: grades[v.GroupPowerId].Grade,
				Exp:   v.Exp,
			},
			Rank: v.Rank,
		})
	}
	if response.MyGroupPower != nil && response.MyGroupPower.Rank == 0 {
		response.MyGroupPower.Rank = 31 // 客户端统一显示30+
	}
	return response, nil
}

func GetGroupPowerStar(model *domain.Model, groupPowerId mysql.ID, _type groupPower_e.GroupPowerStarType, offset, limit int, month string) ([]*group_power_cv.CvGroupPowerStarData, error) {
	rank, err := groupPower_m.GetGroupPowerMonthStarRank(model, groupPowerId, _type, offset, limit, month)
	if err != nil {
		return nil, err
	}
	//var response []group_power_cv.CvGroupPowerStarData
	response := make([]*group_power_cv.CvGroupPowerStarData, 0)
	var userIds []mysql.ID
	for _, row := range rank {
		userIds = append(userIds, row.UserId)
	}
	users, err := user_c.GetUserTinyMap(model, userIds, true)
	//users, err := user_m.GetUserMapByIds(model, userIds)
	for _, row := range rank {
		user := users[row.UserId]
		score := row.Score
		if _type == groupPower_e.GroupPowerStarTypeActive {
			score = score / 60
		}
		if score <= 0 {
			continue
		}
		response = append(response, &group_power_cv.CvGroupPowerStarData{
			User: user_cv.CvUserTiny{
				Id:         user.ID,
				ExternalId: user.ExternalId,
				Code:       user.Code,
				Nick:       user.Nick,
				Avatar:     user.Avatar,
			},
			Score: score,
		})
	}
	return response, nil
}

func CalcGroupPowerMonthRankAct(model *domain.Model) error {
	lastDayTime := time.Now().AddDate(0, 0, -1)
	_now := now.New(lastDayTime)
	calcMonth := _now.BeginningOfMonth().Format(utils.MONTH_FORMAT)
	beginDate, endDate := _now.BeginningOfMonth().Format("2006-01-02"), _now.EndOfMonth().Format("2006-01-02")
	return DoCalcGroupPowerMonthRankAct(model, calcMonth, beginDate, endDate)
}

func DoCalcGroupPowerMonthRankAct(model *domain.Model, calcMonth, beginDate, endDate string) error {
	areaList := []int{1, 2}
	for _, area := range areaList { // 计算获奖
		model := domain.CreateModelContext(model.MyContext)
		response, err := GetGroupPowerRankResp(model, beginDate, endDate, 0, groupPower_e.GroupPowerUserHas, area)
		if err != nil {
			return err
		}
		powerIds := make([]uint64, 0, len(response.Items))
		for i, v := range response.Items {
			powerIds = append(powerIds, v.Id)
			response.Items[i].StarList, err = GetGroupPowerStar(model, v.Id, groupPower_e.GroupPowerStarTypeFamous, 0, 10, time.Now().Format(utils.COMPACT_MONTH_FORMAT))
			if err != nil {
				return err
			}
		}
		// 奖励、日志
		awardList, logList, err := getAwardAndLogList(model, powerIds, response.Items, calcMonth, area)
		if err != nil {
			return err
		}

		err = model.Transaction(func(model *domain.Model) error {
			// log
			err = groupPower_m.CreateActFamilyMonthRankLog(model, logList)
			if err != nil {
				model.Log.Errorf("CalcGroupPowerMonthRankAct area:%d, err:%+v", area, err)
				return err
			}
			// award
			for _, v := range awardList {
				err = award_tx.SendUserAward(model, v, diamond_e.GeneralActivity, msg_e.MgrSendDiamondProperty)
				if err != nil {
					model.Log.Errorf("CalcGroupPowerMonthRankAct award:%+v, err:%+v", v, err)
					return err
				}
			}
			return nil
		})
		if err != nil {
			model.Log.Errorf("CalcGroupPowerMonthRankAct area:%d, err:%+v", area, err)
			return err
		}
	}
	return nil
}

func getAwardAndLogList(model *domain.Model, powerIds []uint64, items []group_power_cv.CvGroupPowerRankData, calcMonth string, area int) ([]*award_tx.UserAward, []*groupPower_m.ActFamilyMonthRankLog, error) {
	// 读奖励配置
	confMap := CalcGroupPowerMonthRankActConf()
	awardList := make([]*award_tx.UserAward, 0, 50) // 奖励
	ownerMap, err := groupPower_m.GetPowerOwnerMap(model, powerIds)
	if err != nil {
		return nil, nil, err
	}
	// log list
	logList := make([]*groupPower_m.ActFamilyMonthRankLog, 0, 50)
	for _, v := range items {
		teamRank := 0
		if v.Rank >= 1 && v.Rank <= 3 {
			teamRank = v.Rank
		} else if v.Rank >= 4 && v.Rank <= 10 {
			teamRank = 4
		} else {
			break
		}
		ownerId, ok := ownerMap[v.Id]
		if !ok || ownerId <= 0 {
			model.Log.Errorf("CalcGroupPowerMonthRankAct 获取势力主错误 familyId:%d", v.Id)
			return nil, nil, bizerr.InvalidParameter
		}
		uRankConfMap, tCOk := confMap[teamRank]
		if !tCOk {
			break
		}
		// 家族长奖励
		oAward, aOk := uRankConfMap[0]
		if !aOk {
			model.Log.Errorf("CalcGroupPowerMonthRankAct 配置错误:%v", confMap)
			return nil, nil, bizerr.InvalidParameter
		}
		oAwardJ, _ := json.Marshal(oAward)
		logList = append(logList, &groupPower_m.ActFamilyMonthRankLog{
			Period:     calcMonth,
			FamilyId:   v.Id,
			RankFamily: v.Rank,
			RankUser:   0,
			UserId:     ownerId,
			Award:      string(oAwardJ),
			Area:       area,
		})
		awardList = append(awardList, &award_tx.UserAward{
			UserId:            ownerId,
			MedalId:           oAward.MedalId,
			MedalDuration:     oAward.MedalDay,
			Diamond:           oAward.Diamond,
			NobleLevel:        oAward.Noble,
			NobleDuration:     oAward.NobleDay,
			HeaddressId:       oAward.HeaddressId,
			HeaddressDuration: oAward.HeaddressDay})
		// 家族成员奖励
		for i, m := range v.StarList {
			uRank := i + 1
			confIdx := 9
			if uRank >= 1 && uRank <= 3 {
				confIdx = 1
			} else if uRank >= 4 && uRank <= 10 {
				confIdx = 4
			}
			mAward, aOk := uRankConfMap[confIdx]
			if !aOk {
				break
			}
			mAwardJ, _ := json.Marshal(mAward)
			logList = append(logList, &groupPower_m.ActFamilyMonthRankLog{
				Period:     calcMonth,
				FamilyId:   v.Id,
				RankFamily: v.Rank,
				RankUser:   uRank,
				UserId:     m.User.Id,
				Award:      string(mAwardJ),
				Area:       area,
			})
			awardList = append(awardList, &award_tx.UserAward{
				UserId:            m.User.Id,
				MedalId:           mAward.MedalId,
				MedalDuration:     mAward.MedalDay,
				Diamond:           mAward.Diamond,
				NobleLevel:        mAward.Noble,
				NobleDuration:     mAward.NobleDay,
				HeaddressId:       mAward.HeaddressId,
				HeaddressDuration: mAward.HeaddressDay})
		}
	}
	return awardList, logList, nil
}

// 奖励配置
type awardConf struct {
	MedalId      uint64
	MedalDay     uint32
	Diamond      uint32
	Noble        uint32
	NobleDay     uint32
	HeaddressId  uint64
	HeaddressDay int
}

func CalcGroupPowerMonthRankActConf() map[int]map[int]*awardConf {
	return map[int]map[int]*awardConf{
		1: {
			0: {MedalId: 5211, MedalDay: 30, Diamond: 320000, Noble: 5, NobleDay: 15},
			1: {MedalId: 5251, MedalDay: 30, Diamond: 120000, Noble: 5, NobleDay: 7, HeaddressId: 2121, HeaddressDay: 30},
			4: {HeaddressId: 2121, HeaddressDay: 30},
		},
		2: {
			0: {MedalId: 5221, MedalDay: 30, Diamond: 220000, Noble: 4, NobleDay: 15},
			1: {MedalId: 5261, MedalDay: 30, Diamond: 100000, Noble: 4, NobleDay: 7, HeaddressId: 2131, HeaddressDay: 30},
			4: {HeaddressId: 2131, HeaddressDay: 30},
		},
		3: {
			0: {MedalId: 5231, MedalDay: 30, Diamond: 120000, Noble: 5, NobleDay: 5},
			1: {MedalId: 5271, MedalDay: 30, Diamond: 70000, HeaddressId: 2141, HeaddressDay: 30},
			4: {HeaddressId: 2141, HeaddressDay: 30},
		},
		4: {
			0: {MedalId: 5241, MedalDay: 30},
			1: {MedalId: 5281, MedalDay: 30, HeaddressId: 2151, HeaddressDay: 30},
		},
	}
}

func GetGroupPowerMaxMemberNum(model *domain.Model, familyId uint64, groupPowerGrade groupPower_e.GroupPowerGrade) (maxNum int) {
	maxNum = group_power_cv.GroupPowerGradePrivilegeNum[groupPowerGrade][0].Num
	ownerId, err := groupPower_m.GetPowerOwner(model.Db, familyId)
	if err == nil {
		//检查是否是贵族
		if flag, err := noble_m.CheckNobleLevel(model.Db, ownerId, 6); err != nil {
		} else if flag {
			newNumMap := map[groupPower_e.GroupPowerGrade]int{
				groupPower_e.GroupPowerGrade0: 400,
				groupPower_e.GroupPowerGrade1: 600,
				groupPower_e.GroupPowerGrade2: 1000,
				groupPower_e.GroupPowerGrade3: 1400}
			if newNum, ok := newNumMap[groupPowerGrade]; ok {
				maxNum = newNum
			}
		}
	}
	users, err := groupPower_m.GetGroupPowerUser(model, familyId)
	if err != nil {
		return
	}
	memberNum := len(users)
	if memberNum > maxNum {
		maxNum = memberNum
	}
	return
}

// 检查需不需要通知用户拉取家族小助手消息
func CheckoutSendMsgToMgr(myContext *mycontext.MyContext, groupPowerId uint64) {
	defer utils.CheckGoPanic()
	model := domain.CreateModelContext(myContext)
	//// 申请人数
	//cnt, err := groupPower_m.CountGroupPowerApply(model, groupPowerId, 0)
	//if err != nil {
	//	model.Log.Errorf("CheckoutSendMsgToMgr err:%v, id:%v", err, groupPowerId)
	//	return
	//}
	//if cnt <= 0 {
	//	return
	//}
	// 找到家族管理员们
	mgrList, err := groupPower_m.GetGroupPowerMgrList(model, groupPowerId)
	if err != nil {
		model.Log.Errorf("CheckoutSendMsgToMgr err:%v, id:%v", err, groupPowerId)
		return
	}
	userMap, err := user_m.GetUserMapByIds(model, mgrList)
	if err != nil {
		model.Log.Errorf("CheckoutSendMsgToMgr err:%v, id:%v", err, groupPowerId)
		return
	}
	for _, v := range mgrList {
		if user, ok := userMap[v]; ok {
			msg_m.SendEmasMsgAssistant(model, user.ExternalId, user.DeviceType)
		}
	}
}
