package game_m

import (
	"git.hilo.cn/hilo-common/domain"
	"gorm.io/gorm"
	"hilo-group/_const/enum/game_e"
	"hilo-group/myerr"
	"time"
)

type GameInfo struct {
	Id             uint64          `json:"id"`
	MgId           string          `json:"mg_id"`
	GameType       game_e.GameType `json:"game_type"`
	Mode           int32           `json:"mode"`
	Piece          int32           `json:"piece"`
	OnOff1         uint8           `json:"on_off1"`
	Diamond        uint64          `json:"diamond"`
	CreateId       uint64          `json:"create_id"`
	Status         uint8           `json:"status"` // '0.未开始 1.游戏中 2.结束'
	TxGroupId      string          `json:"tx_group_id"`
	GameRoundId    string          `json:"game_round_id"`
	BattleStartAt  uint32          `json:"battle_start_at"`
	BattleEndAt    uint32          `json:"battle_end_at"`
	BattleDuration uint32          `json:"battle_duration"`
	AutoMatch      uint8           `json:"auto_match"` // 是否开启自动匹配，0否1是
	CreatedTime    time.Time       `json:"created_time"`
	UpdatedTime    time.Time       `json:"updated_time"`
}

type GameConfig struct {
	Ludo   *GameConfigDiamond `json:"ludo"`
	Uno    *GameConfigDiamond `json:"uno"`
	Domino *GameConfigDiamond `json:"domino"`
}

type GameConfigDiamond struct {
	Diamond []uint32 `json:"diamond"`
}

func GetGameInfo(model *domain.Model, gameType game_e.GameType, gameId uint64, txGroupId, roundId string, battleEndAt, status int64) (*GameInfo, error) {
	res := new(GameInfo)
	db := model.Db
	if gameType != 0 {
		db = db.Where("game_type = ?", gameType)
	}
	if gameId != 0 {
		db = db.Where("id = ?", gameId)
	}
	if txGroupId != "" {
		db = db.Where("tx_group_id = ?", txGroupId)
	}
	if roundId != "" {
		db = db.Where("game_round_id = ?", roundId)
	}
	if battleEndAt != -1 {
		db = db.Where("battle_end_at = ?", battleEndAt)
	}
	if status != -1 {
		db = db.Where("status = ?", status)
	}
	err := db.First(&res).Error
	if err != nil && err != gorm.ErrRecordNotFound {
		return nil, err
	}
	return res, nil
}

// 获取所有未结束的游戏
// return map $txGroupId->[]GameTypes
func GetNotEndGamesMap(model *domain.Model) map[string][]game_e.GameType {
	var res = make(map[string][]game_e.GameType)
	var games []GameInfo
	if err := model.Db.Model(GameInfo{}).Where("status in (0,1)").Find(&games).Error; err != nil {
		model.Log.Errorf("GetNotEndGamesMap fail:%v", err)
		return res
	}
	for _, game := range games {
		res[game.TxGroupId] = []game_e.GameType{game.GameType}
	}
	return res
}

func IsGaming(model *domain.Model, userId uint64, txGroupId string) (bool, error) {
	ids := make([]uint64, 0)
	err := model.DB().Table("game_info").Where("tx_group_id = ? and battle_end_at = 0 and status in (0, 1)", txGroupId).Limit(1).Pluck("id", &ids).Error
	if err != nil {
		return false, myerr.WrapErr(err)
	}
	if len(ids) > 0 {
		userIds := make([]uint64, 0)
		err = model.DB().Table("game_player").Where("game_id = ? and user_id = ? and status in (0,1)", ids[0], userId).Limit(1).Pluck("id", &userIds).Error
		if err != nil {
			return false, myerr.WrapErr(err)
		}
		if len(userIds) > 0 {
			return true, nil
		}
	}
	return false, nil
}
