package jwt

import (
	"git.hilo.cn/hilo-common/resource/config"
	"github.com/dgrijalva/jwt-go"
	"hilo-group/myerr"
	"time"
)

// 载荷，增加用户别名
type Claims struct {
	UserId     uint64
	ExternalId string
	jwt.StandardClaims
}

//生成token
func GenerateToken(userId uint64, externalId string, issuer string) (string, error) {
	jwtConfig := config.GetConfigJWT()
	duration, err := time.ParseDuration(jwtConfig.EXPIRE)
	if err != nil {
		return "", myerr.WrapErr(err)
	}

	expireTime := time.Now().Add(duration)
	claims := Claims{
		UserId:     userId,
		ExternalId: externalId,
		StandardClaims: jwt.StandardClaims{
			ExpiresAt: expireTime.Unix(), //过期时间
			Issuer:    issuer,            //签名的发行者
		},
	}
	tokenClaims := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)
	token, err := tokenClaims.SignedString(GetJWTSecret())
	return token, myerr.WrapErr(err)
}

func GetJWTSecret() []byte {
	return []byte(config.GetConfigJWT().SECRET)
}

//解析token
func ParseToken(token string) (*Claims, error) {
	tokenClaims, err := jwt.ParseWithClaims(token, &Claims{}, func(token *jwt.Token) (interface{}, error) {
		return GetJWTSecret(), nil
	})
	if err != nil {
		return nil, myerr.WrapErr(err)
	}
	if tokenClaims != nil {
		claims, ok := tokenClaims.Claims.(*Claims)
		if ok && tokenClaims.Valid {
			return claims, nil
		}
	}
	return nil, myerr.WrapErr(err)
}
