package service

import (
	"context"
	"encoding/json"
	"fmt"
	"github.com/shopspring/decimal"
	"github.com/spf13/cast"
	"go-micro.dev/v4/errors"
	"gomicro-base/api/member"
	"gomicro-base/api/wallet"
	"gomicro-base/common/errm"
	"gomicro-base/common/logz"
	"gomicro-base/common/validate"
	"gomicro-base/service/wallet/client"
	"gomicro-base/service/wallet/internal/dao"
	"gomicro-base/service/wallet/internal/model"
	"gorm.io/gorm"
	"time"
)

// WalletService 服务层处理逻辑
type WalletService struct {
	dao *dao.Dao
}

func New(d *dao.Dao) *WalletService {
	return &WalletService{
		dao: d,
	}
}

// ChargeOne 单向扣除金币
func (w *WalletService) ChargeOne(ctx context.Context, req *wallet.ChargeOneReq) error {
	err := validate.Struct(req)
	if err != nil {
		return errm.TransParamsError(err)
	}

	// 判断是否冻结
	fm := model.ImsDbPlayFreezeMoney{}
	err = w.dao.DB.Table(fm.TableName()).Where("user_id = ?", req.UserId).Select("id").Take(&fm).Error
	if err != nil && err != gorm.ErrRecordNotFound {
		return err
	}
	// 存在记录
	if fm.Id > 0 {
		return errm.NewMicroError("Gold coin account locked", 401)
	}

	// 调用用户服务修改金额
	_, err = client.GetMemberSvc().SetCredit1(ctx, &member.SetCredit1Req{
		UserId:  req.UserId,
		GoldNum: req.GoldNum * -1, // 减
	})
	if err != nil {
		logz.Error(err)
		return errm.NewMicroError("Insufficient gold coin balance", 403)
	}

	// 写入日志
	payLog := model.ImsDbPlayPaylog{
		UserId:   int(req.UserId),
		PayTime:  int(time.Now().Unix()),
		PayMoney: decimal.NewFromFloat32(req.GoldNum),
		Type:     int(wallet.GoldType_typeCharge),
	}
	err = w.dao.DB.Table(payLog.TableName()).Create(&payLog).Error
	if err != nil {
		logz.Errorf("charge 写入日志失败: %s", err)
		return nil
	}

	return nil
}

// AddGold 发放金币
func (w *WalletService) AddGold(ctx context.Context, req *wallet.AddGoldReq) error {
	// 通过validate tag标签验证参数
	err := validate.Struct(req)
	if err != nil {
		return errm.TransParamsError(err)
	}

	// 调用用户服务修改金额
	_, err = client.GetMemberSvc().SetCredit1(ctx, &member.SetCredit1Req{
		UserId:  req.UserId,
		GoldNum: req.GoldNum, // 加
	})
	if err != nil {
		return errm.NewMicroError(fmt.Sprintf("更新金额失败: %s", errors.FromError(err).Detail))
	}

	nowTime := time.Now().Unix()

	// 写入日志
	payLog := model.ImsDbPlayPaylog{
		UserId:   int(req.UserId),
		PayTime:  int(nowTime),
		PayMoney: decimal.NewFromFloat32(req.GoldNum),
		Type:     int(wallet.GoldType_typeAdd),
	}
	err = w.dao.DB.Table(payLog.TableName()).Create(&payLog).Error
	if err != nil {
		return err
	}

	// 发金币IM消息
	im := model.ImMessage{
		Content: model.ImMessageContent{Money: cast.ToString(req.GoldNum)},
		UserId:  int(req.UserId),
		SendId:  int(req.AdminUserId),
		SysType: "RechargeResult",
	}
	marshal, _ := json.Marshal(im)
	_, err = w.dao.Redis.RPush(ctx, "rongyun_queue", marshal).Result()
	if err != nil {
		return err
	}

	// 写入备注
	r := model.ImsDbPlayPaylogRemark{
		UserId:  int(req.UserId),
		Gold:    cast.ToInt(req.GoldNum),
		PayTime: int(nowTime),
		Remark:  req.Remark,
	}
	return w.dao.DB.Table(r.TableName()).Create(&r).Error
}
