package logz

import (
	"github.com/go-kratos/kratos/v2/log"
	"github.com/spf13/cast"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"os"
)

var (
	_ log.Logger = &zapStd{}

	// kratos映射zap的level
	levelMap = map[log.Level]zapcore.Level{
		log.LevelDebug: zapcore.DebugLevel,
		log.LevelInfo:  zapcore.InfoLevel,
		log.LevelWarn:  zapcore.WarnLevel,
		log.LevelError: zapcore.ErrorLevel,
		log.LevelFatal: zapcore.FatalLevel,
	}
	// DefaultConfig 默认配置
	DefaultConfig = &Config{
		Color: true,
		Level: "DEBUG",
	}
)

type (
	zapStd struct {
		log *zap.SugaredLogger
	}

	Config struct {
		Color bool
		Level string
	}
)

func NewLogger(conf *Config, option ...zap.Option) *zapStd {
	if conf == nil {
		conf = DefaultConfig
	}
	var level zapcore.Level
	if v, ok := levelMap[log.ParseLevel(conf.Level)]; ok {
		level = v
	} else {
		level = zapcore.InfoLevel
	}

	cfg := zap.NewProductionEncoderConfig()
	if conf.Color {
		// 颜色
		cfg.EncodeLevel = zapcore.CapitalColorLevelEncoder
	} else {
		cfg.EncodeLevel = zapcore.CapitalLevelEncoder
	}

	cfg.ConsoleSeparator = "  |  "

	// 指定日志时间格式
	cfg.EncodeTime = zapcore.TimeEncoderOfLayout("2006-01-02 15:04:05.000")
	//cfg.EncodeCaller = zapcore.FullCallerEncoder

	// 使用控制台输出
	encoder := zapcore.NewConsoleEncoder(cfg)
	core := zapcore.NewCore(encoder, zapcore.AddSync(os.Stdout), level)
	l := zap.New(zapcore.NewTee(core), option...)

	zapStdLog := &zapStd{
		log: l.Sugar(),
	}
	return zapStdLog
}

func (z *zapStd) Log(level log.Level, keyvals ...interface{}) error {
	if len(keyvals) == 0 {
		return nil
	}
	if (len(keyvals) & 1) == 1 {
		keyvals = append(keyvals, "key val params invalid")
	}

	var msg string
	kv := make([]interface{}, 0, len(keyvals))

	for i := 0; i < len(keyvals); i += 2 {
		if keyvals[i] == log.DefaultMessageKey {
			msg = cast.ToString(keyvals[i+1])
			continue
		}
		kv = append(kv, keyvals[i], keyvals[i+1])
	}

	switch level {
	case log.LevelDebug:
		z.log.Debugw(msg, kv...)
	case log.LevelInfo:
		z.log.Infow(msg, kv...)
	case log.LevelWarn:
		z.log.Warnw(msg, kv...)
	case log.LevelError:
		z.log.Errorw(msg, kv...)
	case log.LevelFatal:
		z.log.Fatalw(msg, keyvals)
	}

	return nil
}

func (z *zapStd) Flush() {
	z.log.Sync()
}
